package ru.yandex.market.juggler;

import com.google.gson.JsonArray;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.StandardHttpRequestRetryHandler;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 02/08/16
 */
public class JugglerClient implements InitializingBean {

    private static final Logger log = LogManager.getLogger();

    private static final int BATCH_ZIE = 100;

    private String jugglerUrl;
    private int httpTimeoutSeconds = 10;
    private int maxConnections = 10;
    private int retryCount = 3;
    private String userAgent = "Market health-utils java client";

    private HttpClient httpClient;

    @Override
    public void afterPropertiesSet() throws Exception {

        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(httpTimeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .setRetryHandler(new StandardHttpRequestRetryHandler(retryCount, true))
            .setUserAgent(userAgent)
            .build();
    }

    public void sendEvents(List<JugglerEvent> events) throws IOException {
        if (events.isEmpty()) {
            return;
        }
        for (int i = 0; i < Math.min(i + BATCH_ZIE, events.size()); i += BATCH_ZIE) {
            sendBatch(events.subList(i, Math.min(i + BATCH_ZIE, events.size())));
        }

        log.info(events.size() + " events send to juggler");
    }

    private void sendBatch(List<JugglerEvent> events) throws IOException {
        JsonArray eventArray = new JsonArray();
        for (JugglerEvent event : events) {
            eventArray.add(event.toJsonObject());
        }
        HttpPost request = new HttpPost(
            jugglerUrl + "/api/1/batch"
        );
        request.setEntity(new StringEntity(eventArray.toString()));

        try {
            HttpResponse response = executeAndCheckCode(request);
            EntityUtils.consume(response.getEntity());
        } finally {
            request.releaseConnection();
        }
    }

    private HttpResponse executeAndCheckCode(HttpUriRequest request) throws IOException {
        HttpResponse response = httpClient.execute(request);
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
            return response;
        }
        String error = IOUtils.toString(response.getEntity().getContent());

        throw new IOException(
            "Error. Code:" + response.getStatusLine().getStatusCode() +
                ", error: " + error + ", request: '" + request.getURI().toString() + "'"
        );
    }


    public void setHttpTimeoutSeconds(int httpTimeoutSeconds) {
        this.httpTimeoutSeconds = httpTimeoutSeconds;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    public void setRetryCount(int retryCount) {
        this.retryCount = retryCount;
    }

    @Required
    public void setJugglerUrl(String jugglerUrl) {
        this.jugglerUrl = jugglerUrl;
    }

    public void setUserAgent(String userAgent) {
        this.userAgent = userAgent;
    }
}
