package ru.yandex.market.log4j2;

import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.appender.AbstractAppender;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.filter.ThresholdFilter;
import org.apache.logging.log4j.core.layout.HtmlLayout;
import org.apache.logging.log4j.core.net.SmtpManager;
import org.apache.logging.log4j.core.util.Booleans;

import java.io.Serializable;

/**
 * Silent SMPT appender. Send not more then 1 message per hour.
 *
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"/>
 * @date 29.06.2015
 */

@Plugin(name = "SilentSMTP", category = "Core", elementType = "appender")
public class SilentSMTPAppender extends AbstractAppender {

    private static final long serialVersionUID = 1L;
    private static final int DEFAULT_BUFFER_SIZE = 512;
    private static final long DEFAULT_SILENCE_PERIOD_MS = 1000L * 60 * 5;

    private int messageCounter = 0;
    private long lastSentMessageTime = 0L;
    private long silencePeriod = DEFAULT_SILENCE_PERIOD_MS;

    /**
     * The SMTP Manager
     */
    private final SmtpManager manager;

    private SilentSMTPAppender(final String name, final Filter filter, final Layout<? extends Serializable> layout, final SmtpManager manager,
                               final boolean ignoreExceptions) {
        super(name, filter, layout, ignoreExceptions);
        this.manager = manager;
    }

    /**
     * Create a SmtpAppender.
     *
     * @param name          The name of the Appender.
     * @param to            The comma-separated list of recipient email addresses.
     * @param cc            The comma-separated list of CC email addresses.
     * @param bcc           The comma-separated list of BCC email addresses.
     * @param from          The email address of the sender.
     * @param replyTo       The comma-separated list of reply-to email addresses.
     * @param subject       The subject of the email message.
     * @param smtpProtocol  The SMTP transport protocol (such as "smtps", defaults to "smtp").
     * @param smtpHost      The SMTP hostname to send to.
     * @param smtpPortStr   The SMTP port to send to.
     * @param smtpUsername  The username required to authenticate against the SMTP server.
     * @param smtpPassword  The password required to authenticate against the SMTP server.
     * @param smtpDebug     Enable mail session debuging on STDOUT.
     * @param bufferSizeStr How many log events should be buffered for inclusion in the
     *                      message?
     * @param layout        The layout to use (defaults to HtmlLayout).
     * @param filter        The Filter or null (defaults to ThresholdFilter, level of
     *                      ERROR).
     * @param ignore        If {@code "true"} (default) exceptions encountered when appending events are logged; otherwise
     *                      they are propagated to the caller.
     * @return The SmtpAppender.
     */
    @PluginFactory
    public static SilentSMTPAppender createAppender(
        @PluginAttribute("name") final String name,
        @PluginAttribute("to") final String to,
        @PluginAttribute("cc") final String cc,
        @PluginAttribute("bcc") final String bcc,
        @PluginAttribute("from") final String from,
        @PluginAttribute("replyTo") final String replyTo,
        @PluginAttribute("subject") final String subject,
        @PluginAttribute("smtpProtocol") final String smtpProtocol,
        @PluginAttribute("smtpHost") final String smtpHost,
        @PluginAttribute("smtpPort") final String smtpPortStr,
        @PluginAttribute("smtpUsername") final String smtpUsername,
        @PluginAttribute("smtpPassword") final String smtpPassword,
        @PluginAttribute("smtpDebug") final String smtpDebug,
        @PluginAttribute("bufferSize") final String bufferSizeStr,
        @PluginElement("Layout") Layout<? extends Serializable> layout,
        @PluginElement("Filter") Filter filter,
        @PluginAttribute("ignoreExceptions") final String ignore) {
        if (name == null) {
            LOGGER.error("No name provided for SmtpAppender");
            return null;
        }

        final boolean ignoreExceptions = Booleans.parseBoolean(ignore, true);
        final int smtpPort = AbstractAppender.parseInt(smtpPortStr, 0);
        final boolean isSmtpDebug = Boolean.parseBoolean(smtpDebug);
        final int bufferSize = bufferSizeStr == null ? DEFAULT_BUFFER_SIZE : Integer.parseInt(bufferSizeStr);

        if (layout == null) {
            layout = HtmlLayout.createDefaultLayout();
        }
        if (filter == null) {
            filter = ThresholdFilter.createFilter(null, null, null);
        }

        final SmtpManager manager = SmtpManager.getSMTPManager(to, cc, bcc, from, replyTo, subject, smtpProtocol,
            smtpHost, smtpPort, smtpUsername, smtpPassword, isSmtpDebug, filter.toString(), bufferSize);
        if (manager == null) {
            return null;
        }

        return new SilentSMTPAppender(name, filter, layout, manager, ignoreExceptions);
    }

    /**
     * Capture all events in CyclicBuffer.
     *
     * @param event The Log event.
     * @return true if the event should be filtered.
     */
    @Override
    public boolean isFiltered(final LogEvent event) {
        final boolean filtered = super.isFiltered(event);
        if (filtered) {

            manager.add(event);
        }
        return filtered;
    }

    /**
     * Perform SmtpAppender specific appending actions, mainly adding the event
     * to a cyclic buffer and checking if the event triggers an e-mail to be
     * sent.
     *
     * @param event The Log event.
     */
    @Override
    public void append(final LogEvent event) {
        ++messageCounter;

        if (System.currentTimeMillis() - silencePeriod > lastSentMessageTime) {
            manager.sendEvents(getLayout(), event);
            lastSentMessageTime = System.currentTimeMillis();
            messageCounter = 0;
        }

    }
}
