package ru.yandex.market.monitoring;

import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoClientURI;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import com.google.common.base.Strings;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.SimpleMongoDbFactory;
import org.springframework.data.mongodb.core.convert.DefaultMongoTypeMapper;
import org.springframework.data.mongodb.core.convert.MappingMongoConverter;

import java.util.concurrent.TimeUnit;

public class MonitoringMetaDao implements InitializingBean {
    private String mongoUrl;
    private String dbName = "health";
    private String monitoringCollectionName = "monitoring";
    private int connectTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(5);
    private int socketTimeoutMillis = (int) TimeUnit.MINUTES.toMillis(1);
    private boolean ssl;
    private String replicaSet;

    private MongoTemplate mongoTemplate;

    @Override
    public void afterPropertiesSet() throws Exception {
        MongoClientOptions.Builder options = MongoClientOptions.builder()
            .writeConcern(WriteConcern.MAJORITY)
            .readPreference(ReadPreference.primary())
            .connectTimeout(connectTimeoutMillis)
            .socketTimeout(socketTimeoutMillis)
            .sslEnabled(ssl);
        if (!Strings.isNullOrEmpty(replicaSet)) {
            options.requiredReplicaSetName(replicaSet);
        }
        MongoClient mongoClient = new MongoClient(new MongoClientURI(mongoUrl, options));

        mongoTemplate = new MongoTemplate(new SimpleMongoDbFactory(mongoClient, dbName));
        ((MappingMongoConverter) mongoTemplate.getConverter()).setTypeMapper(new DefaultMongoTypeMapper(null));
    }

    public static class MonitoringUnitMeta {
        @Id
        private final String unitName;
        private final long lastUpdateTimeMillis;

        private MonitoringUnitMeta(String unitName, long lastUpdateTimeMillis) {
            this.unitName = unitName;
            this.lastUpdateTimeMillis = lastUpdateTimeMillis;
        }

        String getUnitName() {
            return unitName;
        }

        public long getLastUpdateTimeMillis() {
            return lastUpdateTimeMillis;
        }
    }

    public MonitoringUnitMeta getMonitoringUnitMeta(String unitName) {
        return mongoTemplate.findById(unitName, MonitoringUnitMeta.class, this.monitoringCollectionName);
    }

    public void saveMonitoringUnitMeta(String unitName, long lastUpdateTimeMillis) {
        mongoTemplate.save(new MonitoringUnitMeta(unitName, lastUpdateTimeMillis), this.monitoringCollectionName);
    }

    @Required
    public void setMongoUrl(String mongoUrl) {
        this.mongoUrl = mongoUrl;
    }

    public void setDbName(String dbName) {
        this.dbName = dbName;
    }

    public void setConnectTimeoutMillis(int connectTimeoutMillis) {
        this.connectTimeoutMillis = connectTimeoutMillis;
    }

    public void setSocketTimeoutMillis(int socketTimeoutMillis) {
        this.socketTimeoutMillis = socketTimeoutMillis;
    }

    public void setMonitoringCollectionName(String monitoringCollectionName) {
        this.monitoringCollectionName = monitoringCollectionName;
    }

    public void setSsl(boolean ssl) {
        this.ssl = ssl;
    }

    public void setReplicaSet(String replicaSet) {
        this.replicaSet = replicaSet;
    }
}
