package ru.yandex.market.razladki;

import com.google.common.base.CharMatcher;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 24/08/15
 * https://beta.wiki.yandex-team.ru/Razladki/DizDok/api
 */
public class RazladkiClient implements InitializingBean {

    private static final Logger log = LogManager.getLogger();

    private String launcherUrl;
    private String project;
    private int httpTimeoutSeconds = 60;
    private int maxConnections = 5;

    private HttpClient httpClient;

    @Override
    public void afterPropertiesSet() throws Exception {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(httpTimeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .build();
    }

    public void sendData(List<RazladkiParam> params) throws IOException {
        sendData(params, true);
    }

    public void sendData(List<RazladkiParam> params, boolean override) throws IOException {
        HttpPost request = new HttpPost(
            launcherUrl + "/save_new_data_json/" + project
        );
        JsonObject mainObject = new JsonObject();
        JsonArray dataArray = new JsonArray();
        mainObject.add("override", new JsonPrimitive(override));
        mainObject.add("data", dataArray);
        for (RazladkiParam param : params) {
            JsonObject paramObject = new JsonObject();
            paramObject.add("param", new JsonPrimitive(param.getName()));
            paramObject.add("ts", new JsonPrimitive(param.getTimestampSeconds()));
            paramObject.add("value", new JsonPrimitive(param.getValue()));
            dataArray.add(paramObject);
        }
//        mainObject.add("alarms", new JsonArray());
        request.setEntity(new StringEntity(mainObject.toString())); //TODO stream
        try {
            HttpResponse response = executeAndCheckCode(request);
            EntityUtils.consume(response.getEntity());
        } finally {
            request.releaseConnection();
        }
    }

    private HttpResponse executeAndCheckCode(HttpUriRequest request) throws IOException {
        HttpResponse response = httpClient.execute(request);
        if (response.getStatusLine().getStatusCode() == HttpStatus.SC_OK) {
            return response;
        } else if (response.getStatusLine().getStatusCode() == HttpStatus.SC_CONFLICT) {
            log.warn("Override attempt. Use override flag.");
            return response;
            //TODO
            //https://beta.wiki.yandex-team.ru/Razladki/DizDok/api/#otpravkadannyx
            //https://st.yandex-team.ru/USEREXP-1813
        }
        String error = IOUtils.toString(response.getEntity().getContent());

        throw new IOException(
            "Error. Code:" + response.getStatusLine().getStatusCode() +
                ", error: " + error + ", request: '" + request.getURI().toString() + "'"
        );
    }

    @Required
    public void setProject(String project) {
        this.project = project;
    }

    @Required
    public void setLauncherUrl(String launcherUrl) {
        this.launcherUrl = CharMatcher.is('/').trimFrom(launcherUrl);
    }

    public void setHttpTimeoutSeconds(int httpTimeoutSeconds) {
        this.httpTimeoutSeconds = httpTimeoutSeconds;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

}
