package ru.yandex.market.rotation;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 2019-03-11
 */
public enum ClickhousePartitionType {
    TUPLE(Pattern.compile("tuple\\(\\)"), null),
    TO_YYYYMM(Pattern.compile("toYYYYMM\\((?<partitionByColumn>\\w+)\\)"), getFormatter("yyyyMM")),
    TO_MONDAY(Pattern.compile("toMonday\\((?<partitionByColumn>\\w+)\\)"), getFormatter("yyyy-MM-dd")),
    DATE(Pattern.compile("^(?<partitionByColumn>\\w+)$"), getFormatter("yyyy-MM-dd"));

    Pattern partitionPattern;
    DateTimeFormatter partitionFormat;

    ClickhousePartitionType(Pattern partitionPattern, DateTimeFormatter partitionFormat) {
        this.partitionPattern = partitionPattern;
        this.partitionFormat = partitionFormat;
    }

    public static ClickhousePartitionType define(String partitionBy) {
        return Stream.of(ClickhousePartitionType.values())
            .filter(p -> p.partitionPattern.asPredicate().test(partitionBy))
            .findFirst()
            .orElseThrow(() -> new RuntimeException("Unsupported clickhouse partition expression"));
    }

    public String format(Instant date) {
        switch (this) {
            case TUPLE:
                return "tuple()";
            case TO_YYYYMM:
            case TO_MONDAY:
            case DATE:
                return partitionFormat.format(date);
            default:
                throw new UnsupportedOperationException();
        }
    }

    public Optional<String> getColumnName(String partitionBy) {
        Matcher matcher = this.partitionPattern.matcher(partitionBy);
        if (this != TUPLE && matcher.find()) {
            return Optional.ofNullable(matcher.group("partitionByColumn"));
        }
        return Optional.empty();
    }

    private static DateTimeFormatter getFormatter(String format) {
        return DateTimeFormatter.ofPattern(format).withZone(ZoneId.systemDefault());
    }

    public static String prepareQuotes(String partition) {
        if (Pattern.compile("'?\\d{4}-\\d{2}-\\d{2}'?").asPredicate().test(partition)) {
            return "'" + partition.replace("'", "") + "'";
        } else {
            return partition;
        }
    }

    public static boolean isTuple(String partition) {
        return TUPLE.partitionPattern.asPredicate().test(partition);
    }
}
