package ru.yandex.market.rotation;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;

import ru.yandex.market.clickhouse.ddl.ClickHouseDdlService;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinition;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.TableName;
import ru.yandex.market.clickhouse.ddl.engine.IMergeTreeEngineType;

/**
 * @author Aleksei Malygin <a href="mailto:Malygin-Me@yandex-team.ru"></a>
 * Date: 2019-03-11
 */
public class DataRotationService {
    private final ClickHouseDdlService ddlService;

    public DataRotationService(ClickHouseDdlService ddlService) {
        this.ddlService = ddlService;
    }

    /**
     * @param table              a rotated data table by rotationPeriodDays parameter
     * @param rotationPeriodDays a period in days, required to be more than 0
     * @return {@code DataRotationTask} prepared task, required for further rotation
     * @throws RuntimeException an exception will be thrown when rotationPeriodDays <= 0
     */
    public DataRotationTask findObsoletePartitions(TableName table, int rotationPeriodDays) {
        if (rotationPeriodDays <= 0) {
            throw new RuntimeException(
                String.format("rotationPeriodDays parameter is less than required: '%d' <= 0", rotationPeriodDays)
            );
        }

        String pivotPartition = getPivotPartition(table, rotationPeriodDays);

        return new DataRotationTask(
            getObsoletePartitions(pivotPartition, table),
            rotationPeriodDays,
            pivotPartition
        );
    }

    @VisibleForTesting
    String getPivotPartition(TableName table, int rotationPeriodDays) {
        Instant pivotPartition = Instant.now().minus(rotationPeriodDays, ChronoUnit.DAYS);
        ClickHouseTableDefinition clickHouseTableDefinition = getClickHouseTableDefinition(table);
        String partitionBy = ((IMergeTreeEngineType)clickHouseTableDefinition.getEngine()).getPartitionBy();
        ClickhousePartitionType partitionType = ClickhousePartitionType.define(partitionBy);

        Optional<String> columnName = partitionType.getColumnName(partitionBy);
        if (columnName.isPresent() &&
            ColumnType.Date != clickHouseTableDefinition.getColumn(columnName.get()).getType()) {
            throw new RuntimeException("Unsupported column type for rotation. Supported types: " + ColumnType.Date);
        }
        return partitionType.format(pivotPartition);
    }

    @VisibleForTesting
    ClickHouseTableDefinition getClickHouseTableDefinition(TableName table) {
        return ddlService.getClickHouseTableDefinition(table);
    }

    private Collection<ObsoletePartition> getObsoletePartitions(String pivotPartition, TableName table) {
        Collection<ObsoletePartition> obsoletePartitions = new ArrayList<>();

        if (Strings.isNullOrEmpty(pivotPartition) || ClickhousePartitionType.isTuple(pivotPartition)) {
            return obsoletePartitions;
        }

        List<String> hosts = ddlService.getCluster().getAllHosts();
        String query = String.format(
            "SELECT DISTINCT partition p FROM system.parts" +
                " WHERE active" +
                " AND database = '%s'" +
                " AND table = '%s'" +
                " AND p < '%s'" +
                " ORDER BY p DESC",
            table.getDatabase(),
            table.getTable(),
            pivotPartition
        );

        for (String host : hosts) {
            ddlService.getHostJdbcTemplate(host).query(
                query,
                rs -> {
                    obsoletePartitions.add(
                        new ObsoletePartition(host, table, rs.getString("p"))
                    );
                }
            );
        }
        return obsoletePartitions;
    }

    /**
     * @param task a prepared task {@code DataRotationTask} by
     *             {@link DataRotationService#findObsoletePartitions(TableName, int)}
     * @return {@code Collection<ObsoletePartition>} a list of deleted partitions
     * @throws {@code Exception} an exception could be thrown during unsuccessful delete operation
     */
    public Collection<ObsoletePartition> deleteObsoletePartitions(DataRotationTask task) throws Exception {
        for (ObsoletePartition op : task.getObsoletePartitions()) {
            try {
                ddlService.getHostJdbcTemplate(op.getHost()).update(op.getDropQuery());
            } catch (Exception e) {
                throw new Exception(
                    String.format("Rotation process: cannot drop partition,\n sql:'%s'\n on host:'%s'",
                        op.getDropQuery(),
                        op.getHost()),
                    e
                );
            }
        }
        return task.getObsoletePartitions();
    }
}
