package ru.yandex.market.statface;

import com.google.common.base.Joiner;
import com.google.gson.*;
import com.google.gson.reflect.TypeToken;
import org.apache.commons.io.IOUtils;
import org.apache.http.*;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicHeader;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.market.statface.model.StatfaceComment;

import java.io.IOException;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * https://wiki.yandex-team.ru/statbox/Statface/externalreports/#rabotasapistatface
 *
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 04/03/15
 */
public class StatfaceClient implements InitializingBean {
    private static final Logger log = LogManager.getLogger();

    private static final Charset CHARSET = Charset.forName("UTF-8");
    private static final NameValuePair APPEND_MODE_PARAM = new BasicNameValuePair("_append_mode", "1");
    private static final Header HEADER_CONTENT_TYPE_JSON = new BasicHeader("Content-Type", "application/json");
    private static final int HTTP_2XX = 200;
    private static final int HTTP_3XX = 300;

    private String host = "https://stat-beta.yandex-team.ru/";
    private String uploadHost = host;
    private String user;
    private String password;
    private int timeoutSeconds = 60;
    private int maxConnections = 20;
    private boolean appendMode = true;

    private HttpClient httpClient;

    @Override
    public void afterPropertiesSet() throws Exception {
        int timeoutMillis = (int) TimeUnit.SECONDS.toMillis(timeoutSeconds);

        RequestConfig requestConfig = RequestConfig.custom()
            .setConnectionRequestTimeout(timeoutMillis)
            .setConnectTimeout(timeoutMillis)
            .setSocketTimeout(timeoutMillis)
            .build();

        httpClient = HttpClientBuilder.create()
            .setMaxConnPerRoute(maxConnections)
            .setMaxConnTotal(maxConnections)
            .setDefaultRequestConfig(requestConfig)
            .setDefaultHeaders(Arrays.asList(
                new BasicHeader("StatRobotUser", user),
                new BasicHeader("StatRobotPassword", password)
            ))
            .build();
    }

    /**
     * Загрузка данных.
     *
     * @param report  путь к отчету (проект, путь внутри проекта, указание длинного скейла)
     * @param tsvData Данные в формате tsv
     * @throws IOException
     */
    @Deprecated
    public void sendData(String report, String tsvData) throws IOException {
        HttpPost post = createPostOld(
            report, "add_data",
            //https://wiki.yandex-team.ru/statbox/Statface/externalreports/#obshhierekomentacii
            "tsv_data", tsvData
        );
        executePost(post);
    }


    /*
     curl -v -H "StatRobotAuth: user:password" -F "data=@data.txt" -F "name=Market/Infrastructure/Events/Conductor/minutely" -k 'https://stat-beta.yandex-team.ru/_api/report/data'
      */
    public String sendAnyData(String path, String data) throws URISyntaxException, IOException {
        URIBuilder uriBuilder = new URIBuilder(uploadHost + "/_api/report/data");
        List<NameValuePair> postParameters = new ArrayList<>();
        postParameters.add(new BasicNameValuePair("name", path));
        postParameters.add(new BasicNameValuePair("_request_timeout", Integer.toString(timeoutSeconds)));
        postParameters.add(new BasicNameValuePair("data", data));
        if (appendMode) {
            postParameters.add(APPEND_MODE_PARAM);
        }

        HttpPost request = new HttpPost(uriBuilder.build());
        request.setEntity(new UrlEncodedFormEntity(postParameters, CHARSET));

        return executeRequest(request);
    }

    /**
     * Создание отчета.
     *
     * @param report       путь к отчету (проект, путь внутри проекта, указание длинного скейла)
     * @param reportConfig конфиг отчета (в yaml)
     * @param title        заголовок отчета
     * @throws IOException
     */
    @Deprecated
    public void createReport(String report, String reportConfig, String title) throws IOException {
        HttpPost post = createPostOld(
            report, "create_report",
            "title", title,
            "cube_config", reportConfig
        );
        executePost(post);
    }

    public void sendData(StatfaceData statfaceData) throws IOException {
        if (statfaceData.isEmpty()) {
            log.warn("Empty data for report: " + statfaceData.getReportName());
            return;
        }
        HttpPost post = createPost(
            "/_api/report/data", true,
            "name", statfaceData.getReportName(),
            "scale", statfaceData.getPeriod().getShortName(),
            "json_data", statfaceData.toJson()
        );
        executePost(post);
    }

    /**
     * Создание отчета.
     *
     * @param config Конфигурация отчета
     * @throws IOException
     */
    public void createReport(StatfaceReportConfiguration config) throws IOException {
//        HttpPost post = new HttpPost(host + "/" + "_api/report/config");
//        post.setEntity(new StringEntity(config.toJson()));
        HttpPost post = createPost(
            "_api/report/config", false,
            "json_config", config.getJsonConfigObject().toString(),
            "name", config.getName(),
            "scale", config.getPeriod().getShortName(),
            "verbose", "1"
        );
        executePost(post);
    }

    public void truncateReport(StatfaceReportConfiguration config) throws IOException {
        truncateReport(config.getName(), config.getPeriod());
    }

    public void truncateReport(String name, StatfacePeriod period) throws IOException {
        HttpPost post = createPost(
            "_api/report/truncate", false,
            "name", name,
            "scale", period.getShortName()
        );
        executePost(post);
    }

    public void deleteReport(String name) throws IOException {
        HttpPost post = createPost(
            "_api/report/delete_report", false,
            "name", name
        );
        executePost(post);
    }


    /**
     * Получение комментариев.
     * https://wiki.yandex-team.ru/statbox/statface/Comments/#get
     *
     * @param path    путь к отчету (проект, путь внутри проекта, указание длинного скейла)
     * @param dateMin Начало отрезка времени
     * @param dateMax Конец отрезка времени
     * @param fields  список имён полей, `none` – специальное значение элемента списка
     * @param tags    выбрать комментарии у которых есть хотя бы один из этих тэгов
     * @throws URISyntaxException
     */
    public List<StatfaceComment> getComments(String path, Date dateMin, Date dateMax,
                                             List<String> fields, List<String> tags)
        throws URISyntaxException, IOException {

        URIBuilder builder = new URIBuilder(host + "/_v3/comments/?");
        builder.addParameter("path", path);
        builder.addParameter("date_min", String.valueOf(dateMin.getTime()));
        if (dateMax != null) {
            builder.addParameter("date_max", String.valueOf(dateMax.getTime()));
        }
        if (fields != null && !fields.isEmpty()) {
            builder.addParameter("field_name", Joiner.on(",").skipNulls().join(fields));
        }
        if (tags != null && !tags.isEmpty()) {
            builder.addParameter("tags", Joiner.on(",").skipNulls().join(tags));
        }

        String result = executeRequest(new HttpGet(builder.build()));
        log.debug(result);
        return deserializeComments(result);
    }

    private static final JsonParser JSON_PARSER = new JsonParser();

    GsonBuilder gsonBuilder = new GsonBuilder()
        .setFieldNamingPolicy(FieldNamingPolicy.LOWER_CASE_WITH_UNDERSCORES)
        .setDateFormat("yyyy-MM-dd'T'HH:mm:ss")
        .setPrettyPrinting();

    protected List<StatfaceComment> deserializeComments(String data) {
        JsonElement element = JSON_PARSER.parse(data);
        return deserializeCommentsArray(element.getAsJsonObject().getAsJsonArray("comments"));
    }

    protected List<StatfaceComment> deserializeCommentsArray(JsonArray comments) {
        Gson gson = gsonBuilder.create();
        Type collectionType = new TypeToken<List<StatfaceComment>>() {
        }.getType();
        return gson.fromJson(comments, collectionType);
    }

    protected List<StatfaceComment> deserializeCommentsArray(String comments) {
        Gson gson = gsonBuilder.create();
        Type collectionType = new TypeToken<List<StatfaceComment>>() {
        }.getType();
        return gson.fromJson(comments, collectionType);
    }

    /**
     * Установка комментария.
     * https://wiki.yandex-team.ru/statbox/statface/Comments/#post
     * <p>
     * Важно: на 5ти минутном графике время комментария также должно быть кратно 5ти минутам.
     */
    public List<StatfaceComment> setComment(StatfaceComment comment) throws URISyntaxException, IOException {
        HttpPost request = new HttpPost(new URI(host + "/_v3/comments/?"));
        request.addHeader(HEADER_CONTENT_TYPE_JSON);
        request.setEntity(new StringEntity(gsonBuilder.create().toJson(comment)));
        String result = executeRequest(request);
        log.debug(result);
        return deserializeCommentsArray(result);
    }

    public List<StatfaceComment> setComment(String path, Date date, String subject)
        throws URISyntaxException, IOException {
        return setComment(new StatfaceComment(path, date, subject));
    }

    public void deleteComment(long id) throws IOException {
        HttpPost request = new HttpPost(host + "/_v3/comments/?_method=DELETE&id=" + id);
        executeRequest(request);
    }

    private String executeRequest(HttpRequestBase request) throws IOException {
        try {
            log.debug(">>> Statface request: " + request.getURI());
            HttpResponse response = httpClient.execute(request);
            int httpCode = response.getStatusLine().getStatusCode();
            if (httpCode < HTTP_2XX || httpCode >= HTTP_3XX) {
                String message = IOUtils.toString(response.getEntity().getContent());
                throw new IOException(String.format("Wrong http code: %s, message: %s at %s",
                    httpCode, message, request.getURI()));
            }
            HttpEntity entity = response.getEntity();
            return entity != null ? EntityUtils.toString(entity) : null;
        } finally {
            request.releaseConnection();
        }
    }

    private void executePost(HttpPost post) throws IOException {
        try {
            HttpResponse response = httpClient.execute(post);
            int httpCode = response.getStatusLine().getStatusCode();
            if (httpCode != HttpStatus.SC_OK) {
                String message = IOUtils.toString(response.getEntity().getContent());
                throw new IOException("Wrong http code:" + httpCode + ", message: " + message);
            }
        } finally {
            post.releaseConnection();
        }
    }

    private HttpPost createPost(String method, boolean sendData, String... params) {
        HttpPost post = new HttpPost(host + "/" + method);
        if (params.length % 2 != 0) {
            throw new IllegalArgumentException();
        }
        ArrayList<NameValuePair> postParameters = new ArrayList<>(params.length / 2 + 2);
        postParameters.add(new BasicNameValuePair("_request_timeout", Integer.toString(timeoutSeconds)));
        if (sendData && appendMode) {
            postParameters.add(APPEND_MODE_PARAM);
        }
        for (int i = 0; i < params.length; i += 2) {
            postParameters.add(new BasicNameValuePair(params[i], params[i + 1]));
        }
        post.setEntity(new UrlEncodedFormEntity(postParameters, CHARSET));
        return post;
    }

    @Deprecated
    private HttpPost createPostOld(String report, String method, String... params) {
        HttpPost post = new HttpPost(host + "/" + report);
        post.addHeader(method, method);
        if (params.length % 2 != 0) {
            throw new IllegalArgumentException();
        }
        ArrayList<NameValuePair> postParameters = new ArrayList<>(params.length / 2 + 2);
        postParameters.add(new BasicNameValuePair("cmd", method));
        postParameters.add(new BasicNameValuePair("_request_timeout", Integer.toString(timeoutSeconds)));
        if (appendMode && method.equals("add_data")) {
            postParameters.add(APPEND_MODE_PARAM);
        }
        for (int i = 0; i < params.length; i += 2) {
            postParameters.add(new BasicNameValuePair(params[i], params[i + 1]));
        }
        post.setEntity(new UrlEncodedFormEntity(postParameters, CHARSET));
        return post;
    }

    public <K, V> void createDictionary(Map<K, V> data, String dictName) throws IOException {
        if (data == null || data.isEmpty()) {
            log.warn("empty dictionary, nothing to add");
            return;
        }
        Type collectionType = new TypeToken<Map<K, V>>() {
        }.getType();
        HttpPost post = createPost(
            "/_api/dictionary", false,
            "name", dictName,
            "language", "",
            "dictionary", gsonBuilder.create().toJson(data, collectionType)
        );
        executePost(post);
    }

    public <K, V> Map<K, V> getDictionary(String dictName) throws URISyntaxException, IOException {
        URIBuilder builder = new URIBuilder(host + "/_api/dictionary/?");
        builder.addParameter("name", dictName);
        HttpGet httpGet = new HttpGet(builder.build());
        Type collectionType = new TypeToken<Map<K, V>>() {
        }.getType();
        return gsonBuilder.create().fromJson(executeRequest(httpGet), collectionType);
    }

    public void deleteDictionary(String dictName) throws URISyntaxException, IOException {
        URIBuilder builder = new URIBuilder(host + "/_api/dictionary/?");
        builder.addParameter("name", dictName);
        HttpDelete httpDelete = new HttpDelete(builder.build());
        executeRequest(httpDelete);
    }

    public void setTimeoutSeconds(int timeoutSeconds) {
        this.timeoutSeconds = timeoutSeconds;
    }

    public void setMaxConnections(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    public void setHost(String host) {
        this.host = host;
    }

    public void setUploadHost(String uploadHost) {
        this.uploadHost = uploadHost;
    }

    @Required
    public void setUser(String user) {
        this.user = user;
    }

    @Required
    public void setPassword(String password) {
        this.password = password;
    }

    public void setAppendMode(boolean appendMode) {
        this.appendMode = appendMode;
    }
}