package ru.yandex.market.statface;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 11/02/16
 */
public class StatfaceField {

    private final String name;
    private final boolean isDimension;
    private final DataType dataType;
    private final String title;
    private final ViewType viewType;
    private final Integer precision;

    public static final StatfaceField DATE_FIELD = new StatfaceField(
        "fielddate", true, StatfaceField.DataType.date, "fielddate"
    );

    public StatfaceField(String name, boolean isDimension, DataType dataType, String title) {
        this(name, isDimension, dataType, title, null, null);
    }

    public StatfaceField(String name, boolean isDimension, DataType dataType, String title,
                         ViewType viewType, Integer precision) {
        if (!isDimension && !dataType.canBeMeasure()) {
            throw new IllegalArgumentException(dataType.name() + " must be dimension");
        }
        if (viewType != null && !viewType.hasPrecision() && precision != null) {
            throw new IllegalArgumentException(String.format("Field '%s' contains precision, " +
                    "which is not allowed for view type '%s'", name, viewType));
        }

        this.name = name;
        this.isDimension = isDimension;
        this.dataType = dataType;
        this.title = title;
        this.viewType = viewType;
        this.precision = precision;
    }

    public String getName() {
        return name;
    }

    public String getTitle() {
        return title;
    }

    public ViewType getViewType() {
        return viewType;
    }

    public boolean hasPrecision() {
        return precision != null;
    }

    public Integer getPrecision() {
        return precision;
    }

    public boolean isDimension() {
        return isDimension;
    }

    public DataType getDataType() {
        return dataType;
    }

    public enum DataType {
        date,
        tree,
        string,
        number;

        private String getStatfaceName() {
            return name().toLowerCase();
        }

        private boolean canBeMeasure() {
            return this != tree;
        }
    }

    public enum ViewType {
        Boolean(false),
        BytesSize(false),
        BytesSpeed(true),
        Float(true),
        Integer(false),
        Number(true),
        Percentage(true),
        Percent(true),
        RoundedFloat(true),
        String(false),
        URL(false),
        Selector(false);

        private boolean allowsPrecision = false;

        ViewType(boolean allowsPrecision) {
            this.allowsPrecision = allowsPrecision;
        }

        public boolean hasPrecision() {
            return this.allowsPrecision;
        }
    }
}
