package ru.yandex.market.statface;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;

import java.util.List;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 10/02/16
 */
public class StatfaceReportConfiguration {

    private final String name;
    private final String title;
    private final StatfacePeriod period;
    private final List<StatfaceField> fields;
    private final List<StatfaceGraph> graphs;

    private boolean ytOnly = false;


    public StatfaceReportConfiguration(String name, String title, StatfacePeriod period,
                                       List<StatfaceField> fields, List<StatfaceGraph> graphs) {
        this.name = name;
        this.title = title;
        this.period = period;
        this.fields = fields;
        this.graphs = graphs;
    }

    public StatfaceReportConfiguration(String name, String title,
                                       StatfacePeriod period, List<StatfaceField> fields) {
        this(name, title, period, fields, null);
    }


    public String toJson() {
        return getJsonConfigObject().toString();
    }

    public JsonObject getJsonConfigObject() {
        JsonObject jsonConfigObject = new JsonObject();
        jsonConfigObject.addProperty("title", title);
        jsonConfigObject.add("user_config", getUserConfigObject());
        return jsonConfigObject;
    }

    public JsonObject getUserConfigObject() {
        JsonObject userConfigObject = new JsonObject();

        userConfigObject.add("dimensions", getFieldsJsonArray(true));
        userConfigObject.add("measures", getFieldsJsonArray(false));
        userConfigObject.add("titles", getTitlesJsonObject());
        if (graphs != null) {
            userConfigObject.add("graphs", getGraphsJsonArray());
        }
        userConfigObject.add("view_types", getViewTypesJsonObject());
        JsonObject cubeObject = getCubeObject();
        if (cubeObject != null) {
            userConfigObject.add("cube", getCubeObject());
        }

        return userConfigObject;
    }

    private JsonObject getViewTypesJsonObject() {
        JsonObject viewTypesObject = new JsonObject();
        for (StatfaceField field : fields) {
            if (field.getViewType() == null) {
                continue;
            }
            JsonObject fieldViewTypeObject = new JsonObject();
            fieldViewTypeObject.addProperty("type", field.getViewType().name());
            if (field.hasPrecision()) {
                fieldViewTypeObject.addProperty("precision", field.getPrecision());
            }
            viewTypesObject.add(field.getName(), fieldViewTypeObject);
        }
        return viewTypesObject;
    }

    public JsonArray getFieldsJsonArray(boolean dimension) {
        JsonArray fieldsArray = new JsonArray();
        for (StatfaceField field : fields) {
            if (field.isDimension() == dimension) {
                JsonObject fieldObject = new JsonObject();
                fieldObject.addProperty(field.getName(), field.getDataType().name());
                fieldsArray.add(fieldObject);
            }
        }
        return fieldsArray;
    }

    public JsonObject getTitlesJsonObject() {
        JsonObject titleObject = new JsonObject();
        for (StatfaceField field : fields) {
            if (field.getTitle() != null) {
                titleObject.addProperty(field.getName(), field.getTitle());
            }
        }
        return titleObject;
    }

    public JsonArray getGraphsJsonArray() {
        JsonArray graphsArray = new JsonArray();
        for (StatfaceGraph graph : graphs) {
            JsonObject graphObject = new JsonObject();
            if (graph.getTitle() != null) {
                graphObject.addProperty("title", graph.getTitle());
            }
            if (graph.getType() != null) {
                graphObject.addProperty("type", graph.getType().name());
            }
            JsonArray graphFieldsArray = new JsonArray();
            for (String field : graph.getFields()) {
                graphFieldsArray.add(new JsonPrimitive(field));
            }
            graphObject.add("fields", graphFieldsArray);
            graphsArray.add(graphObject);
        }
        return graphsArray;
    }

    private JsonObject getCubeObject() {
        JsonObject cubeObject = new JsonObject();
        if (!ytOnly) {
            return null;
        }
        cubeObject.addProperty("yt_only", 1);
        return cubeObject;
    }

    public void setYtOnly(boolean ytOnly) {
        this.ytOnly = ytOnly;
    }

    public String getName() {
        return name;
    }

    public String getTitle() {
        return title;
    }

    public StatfacePeriod getPeriod() {
        return period;
    }

    public List<StatfaceField> getFields() {
        return fields;
    }

    public List<StatfaceGraph> getGraphs() {
        return graphs;
    }

    public boolean isYtOnly() {
        return ytOnly;
    }


}
