package ru.yandex.market.logshatter.parser;

import org.openjdk.jmh.annotations.Benchmark;
import org.openjdk.jmh.annotations.BenchmarkMode;
import org.openjdk.jmh.annotations.Fork;
import org.openjdk.jmh.annotations.Level;
import org.openjdk.jmh.annotations.Measurement;
import org.openjdk.jmh.annotations.Mode;
import org.openjdk.jmh.annotations.OutputTimeUnit;
import org.openjdk.jmh.annotations.Param;
import org.openjdk.jmh.annotations.Scope;
import org.openjdk.jmh.annotations.Setup;
import org.openjdk.jmh.annotations.State;
import org.openjdk.jmh.annotations.TearDown;
import org.openjdk.jmh.annotations.Warmup;
import org.openjdk.jmh.runner.Runner;
import org.openjdk.jmh.runner.RunnerException;
import org.openjdk.jmh.runner.options.Options;
import org.openjdk.jmh.runner.options.OptionsBuilder;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.useragent.FakeUserAgentDetector;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

/**
 * @author Dmitry Poluyanov <a href="https://t.me/neiwick">Dmitry Poluyanov</a>
 * @since 31.07.17
 */
@BenchmarkMode(Mode.Throughput)
@OutputTimeUnit(TimeUnit.MILLISECONDS)
@State(Scope.Benchmark)
@Fork(value = 2, jvmArgsAppend = {"-server"/*, "-XX:+PrintCompilation"*/})
@Warmup(iterations = 5)
@Measurement(iterations = 10)
public class ParserContextImplBenchmark {
    private ParserContextImpl parserContext;
    private Date date;
    private int[] columnValues;
    private LogBatch logBatch;

    @Param({"0.0", "0.05", "1.0"})
    private float sampleRatio = 0.05f;

    // todo exclude META-INF/BenchmarkList from iceberg-inside libs
    public static void main(String[] args) throws RunnerException {
        Options opt = new OptionsBuilder()
            .include(ParserContextImplBenchmark.class.getSimpleName())
            .warmupIterations(3)
            .measurementIterations(5)
            .mode(Mode.Throughput)
            .timeUnit(TimeUnit.MILLISECONDS)
            .forks(1)
            .build();

        new Runner(opt).run();
    }

    @Setup(Level.Iteration)
    public void setUp() {
        date = new Date();
        columnValues = new int[]{0, 100, 500, 10000, Integer.MAX_VALUE};

        List<Column> columns = new ArrayList<>(7);
        columns.add(new Column("date", ColumnType.Date));
        columns.add(new Column("timestamp", ColumnType.DateTime));

        for (int i = 0; i < 5; i++) {
            columns.add(new Column(String.valueOf(i), ColumnType.Int32));
        }

        logBatch = new LogBatch(
            Stream.empty(),
            0,
            0,
            0,
            Duration.ofMillis(0),
            columns,
            "sourceName"
        );
        parserContext = new ParserContextImpl(logBatch, null, null, false, sampleRatio, new FakeUserAgentDetector());
    }

    @TearDown(Level.Iteration)
    public void tearDown() {
        logBatch = null;
        parserContext = null;
        date = null;
        columnValues = null;
        // force gc, for release allocated resources
        System.gc();
    }

    @Benchmark
    public void writeMeasure() {
        parserContext.write(date, columnValues[0], columnValues[1], columnValues[2],
            columnValues[3], columnValues[4]);
    }

//    @Fork(value = 1, jvmArgsAppend = {"-server"/*, "-XX:+PrintCompilation"*/})
//    @Warmup(iterations = 3)
//    @Measurement(iterations = 5)
//    @BenchmarkMode(Mode.AverageTime)
//    @OutputTimeUnit(TimeUnit.NANOSECONDS)
//    public static class CurrentTimeMillis {
//        @Benchmark
//        public long currentTimeMillis() {
//            return System.currentTimeMillis();
//        }
//
//        @Benchmark
//        public float threadLocalRandom() {
//            return ThreadLocalRandom.current().nextFloat() * 0.05f;
//        }
//    }
}
