package ru.yandex.home.logshatter.parser;

import com.google.common.primitives.UnsignedLong;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.*;

import java.util.Arrays;
import java.util.Date;



/**
 * @author Evgeny Bogdanov <a href="mailto:evbogdanov@yandex-team.ru"></a>
 * @date 30/03/18
 */
public class HomeAccessLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.create(
        new MergeTree(
            "toYYYYMMDD(" + TableDescription.DATE_COLUMN.getName() + ")",
            Arrays.asList("yandexuid")
        ),
        Arrays.asList(
            new Column("host", ColumnType.String),
            new Column("method", ColumnType.String),
            new Column("requestid", ColumnType.String),
            new Column("yandexuid", ColumnType.UInt64),
            new Column("icookie", ColumnType.UInt64),
            new Column("yuid_days", ColumnType.UInt32),
            new Column("hostname", ColumnType.String),
            new Column("timezone", ColumnType.String),
            new Column("x_forwarded_for", ColumnType.String),
            new Column("request", ColumnType.String),
            new Column("https", ColumnType.UInt8),
            new Column("m_zone", ColumnType.String),
            new Column("pid", ColumnType.UInt32),
            new Column("referer", ColumnType.String),
            new Column("status", ColumnType.UInt16),
            new Column("user_agent", ColumnType.String),
            new Column("vhost", ColumnType.String),
            new Column("timing", ColumnType.String),
            new Column("wait", ColumnType.Float32),
            new Column("wait_avg", ColumnType.Float32),
            new Column("ua.browserengine", ColumnType.String),
            new Column("ua.browserengineversion", ColumnType.String),
            new Column("ua.browsername", ColumnType.String),
            new Column("ua.browserversion", ColumnType.String),
            new Column("ua.osfamily", ColumnType.String),
            new Column("ua.osversion", ColumnType.String),
            new Column("ua.ismobile", ColumnType.UInt8),
            new Column("ua.istablet", ColumnType.UInt8),
            new Column("ua.istv", ColumnType.UInt8),
            new Column("test_buckets", ColumnType.ArrayUInt32),
            new Column("enabled_test_buckets", ColumnType.ArrayUInt32),
            new Column("exp", ColumnType.ArrayString),
            new Column("login", ColumnType.String),
            new Column("geo_h", ColumnType.String),
            new Column("geo_h_age", ColumnType.String),
            new Column("geo_h_loc", ColumnType.String),
            new Column("geo_h_region", ColumnType.String),
            new Column("geo_region", ColumnType.String),
            new Column("m_content", ColumnType.String)
        )
    );

    static Integer[] parseTestBuckets(String inputString) {
        if (inputString == null || inputString.isEmpty()) {
            return new Integer[0];
        }

        String[] testBucketsList = inputString.split(";");
        Integer testBucketsSize = testBucketsList.length;
        if (testBucketsList[0].isEmpty()) {
            testBucketsSize = 0;
        }
        Integer[] testBuckets = new Integer[testBucketsSize];
        for (int i = 0; i < testBucketsList.length; i++) {
            int splitIndex = testBucketsList[i].indexOf(',');
            if (splitIndex > 0) {
                testBuckets[i] = Integer.valueOf(testBucketsList[i].substring(0, splitIndex));
            }
        }
        return testBuckets;
    }

    private static String parseLogin(String inputStrin) {
        String login = "";
        String[] cookies = inputStrin.split(";");
        for (String cookie : cookies) {
            if (cookie.startsWith("yandex_login=") || cookie.startsWith(" yandex_login=")) {
                login = cookie.split("=")[1];
            }
        }
        return login;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        TskvSplitter tskvSplitter = new TskvSplitter(line);
        Date date = ParseUtils.parseDateInSeconds(tskvSplitter.getString("unixtime"));
        String requestid = tskvSplitter.getOptionalString("requestid", "");

        String host = tskvSplitter.getOptionalString("host", "");
        String method = tskvSplitter.getOptionalString("method", "");
        UnsignedLong yandexUid = tskvSplitter.getOptionalUnsignedLong("yandexuid", UnsignedLong.valueOf(0L));
        UnsignedLong icookie = tskvSplitter.getOptionalUnsignedLong("icookie", UnsignedLong.valueOf(0L));

        Integer yuidDays = tskvSplitter.getOptionalInt("yuid_days", 0);
        String hostname = tskvSplitter.getOptionalString("hostname", "");
        String timezone = tskvSplitter.getOptionalString("timezone", "");
        String xForwardedFor = tskvSplitter.getOptionalString("x_forwarded_for", "");
        String request = tskvSplitter.getOptionalString("request", "");
        Integer https = tskvSplitter.getOptionalInt("https", 0);
        String mZone = tskvSplitter.getOptionalString("m_zone", "");
        Integer pid = tskvSplitter.getOptionalInt("pid", 0);
        String referer = tskvSplitter.getOptionalString("referer", "");
        Integer status = tskvSplitter.getOptionalInt("status", 0);
        String userAgent = tskvSplitter.getOptionalString("user_agent", "");
        String vhost = tskvSplitter.getOptionalString("vhost", "");
        String timing = tskvSplitter.getOptionalString("timing", "");
        Float wait = new Float(tskvSplitter.getOptionalDouble("wait", 0.0));
        Float waitAvg = new Float(tskvSplitter.getOptionalDouble("wait_avg", 0.0));
        String uaBrowserengine = tskvSplitter.getOptionalString("ua.browserengine", "");
        String uaBrowserengineversion = tskvSplitter.getOptionalString("ua.browserengineversion", "");
        String uaBrowsername = tskvSplitter.getOptionalString("ua.browsername", "");
        String uaBrowserversion = tskvSplitter.getOptionalString("ua.browserversion", "");
        String uaOsfamily = tskvSplitter.getOptionalString("ua.osfamily", "");
        String uaOsversion = tskvSplitter.getOptionalString("ua.osversion", "");
        Integer uaIsmobile = tskvSplitter.getOptionalInt("ua.ismobile", 0);
        Integer uaIstablet = tskvSplitter.getOptionalInt("ua.istablet", 0);
        Integer uaIstv = tskvSplitter.getOptionalInt("ua.istv", 0);
        String mContent = tskvSplitter.getOptionalString("m_content", "");

        Integer[] testBuckets = parseTestBuckets(tskvSplitter.getOptionalString("test-bucket", ""));
        Integer[] enabledTestBuckets =
            parseTestBuckets(tskvSplitter.getOptionalString("enabled-test-buckets", ""));

        String[] exp = tskvSplitter.getOptionalString("exp", "").split(",");
        String login = parseLogin(tskvSplitter.getOptionalString("cookies", ""));
        String geoH = tskvSplitter.getOptionalString("geo_h", "");
        String geoHAge = tskvSplitter.getOptionalString("geo_h_age", "");
        String geoHLoc = tskvSplitter.getOptionalString("geo_h_loc", "");
        String geoHRegion = tskvSplitter.getOptionalString("geo_h_region", "");
        String geoRegion = tskvSplitter.getOptionalString("geo_region", "");

        context.write(date, host, method, requestid, yandexUid, icookie, yuidDays, hostname, timezone,
            xForwardedFor, request,
            https, mZone,
            pid, referer, status, userAgent, vhost, timing, wait, waitAvg, uaBrowserengine,
            uaBrowserengineversion, uaBrowsername, uaBrowserversion, uaOsfamily, uaOsversion,
            uaIsmobile, uaIstablet, uaIstv, testBuckets, enabledTestBuckets,
            exp, login, geoH, geoHAge, geoHLoc, geoHRegion, geoRegion, mContent);
    }


    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
