package ru.yandex.home.logshatter.parser;

import com.google.common.base.Splitter;
import com.google.common.primitives.UnsignedLong;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TskvSplitter;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;
import java.util.Date;


/**
 * @author Evgeny Bogdanov <a href="mailto:evbogdanov@yandex-team.ru"></a>
 * @date 20/11/17
 */
public class HomeBlockdisplayLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.create(
        new MergeTree(
            "toYYYYMMDD(" + TableDescription.DATE_COLUMN.getName() + ")",
            Arrays.asList("yandexuid")
        ),
        Arrays.asList(
            new Column("show_id", ColumnType.String),
            new Column("yandexuid", ColumnType.UInt64),
            new Column("icookie", ColumnType.UInt64),
            new Column("test_buckets", ColumnType.ArrayUInt32),
            new Column("enabled_test_buckets", ColumnType.ArrayUInt32),
            new Column("blocks", ColumnType.ArrayString)
        )
    );

    private static Integer[] parseTestBuckets(String inputString) {
        return HomeAccessLogParser.parseTestBuckets(inputString);
    }

    private static String[] copyStringArray(String[] original, int newLength) {
        String[] copy = new String[Math.min(original.length, newLength)];
        System.arraycopy(original, 0, copy, 0, Math.min(original.length, newLength));
        return copy;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        TskvSplitter tskvSplitter = new TskvSplitter(line);
        Date date = ParseUtils.parseDateInSeconds(tskvSplitter.getString("unixtime"));
        String show_id = tskvSplitter.getString("show_id");
        UnsignedLong yandexUid = tskvSplitter.getUnsignedLong("yandexuid");
        UnsignedLong icookie = tskvSplitter.getOptionalUnsignedLong("icookie", UnsignedLong.valueOf(0L));

        Integer[] testBuckets = parseTestBuckets(tskvSplitter.getOptionalString("test-bucket", ""));
        Integer[] enabledTestBuckets =
            parseTestBuckets(tskvSplitter.getOptionalString("enabled-test-buckets", ""));

        String blocks = tskvSplitter.getOptionalString("blocks", "");
        String[] blocksSplitted = blocks.split("\\\\t");
        Set<String> blocksForSave = new HashSet<>();
        int i = 0;
        while (i < blocksSplitted.length) {
            blocksForSave.add(String.join(".", copyStringArray(blocksSplitted[i].split("\\."), 5)));
            i++;
            Integer parameters = Integer.valueOf(blocksSplitted[i]);
            i += parameters + 1;
        }

        context.write(date, show_id, yandexUid, icookie, testBuckets, enabledTestBuckets, blocksForSave.toArray());
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
