package ru.yandex.home.logshatter.parser;

import com.google.common.base.Splitter;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.common.primitives.UnsignedLong;

import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Date;
import java.util.Map;
import java.util.HashMap;
import java.util.ArrayList;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;


/**
 * @author Evgeny Bogdanov <a href="mailto:evbogdanov@yandex-team.ru"></a>
 * @date 20/11/17
 */
public class HomeRedirLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.create(
        new MergeTree(
            "toYYYYMMDD(" + TableDescription.DATE_COLUMN.getName() + ")",
            Arrays.asList("yandexuid")
        ),
        Arrays.asList(
            new Column("at", ColumnType.Int8),
            new Column("event_type", ColumnType.String),
            new Column("show_host", ColumnType.String),
            new Column("path", ColumnType.String),
            new Column("referer", ColumnType.String),
            new Column("reqid", ColumnType.String),
            new Column("url", ColumnType.String),
            new Column("yandexuid", ColumnType.UInt64),
            new Column("icookie", ColumnType.UInt64),
            new Column("timefrs", ColumnType.UInt32),
            new Column("timefs", ColumnType.UInt32),
            new Column("testids", ColumnType.ArrayUInt32),
            new Column("ua.browserengine", ColumnType.String),
            new Column("ua.browserengineversion", ColumnType.String),
            new Column("ua.browsername", ColumnType.String),
            new Column("ua.browserversion", ColumnType.String),
            new Column("ua.osfamily", ColumnType.String),
            new Column("ua.osversion", ColumnType.String),
            new Column("ua.ismobile", ColumnType.UInt8),
            new Column("ua.istablet", ColumnType.UInt8),
            new Column("ua.istv", ColumnType.UInt8)
        )
    );

    private static void getCountersWithURLs(JsonElement block, String path, List<String> output) {
        path = path + '.' + block.getAsJsonObject().get("ctag").getAsString();
        if (block.getAsJsonObject().has("children")) {
            for (JsonElement child_block : block.getAsJsonObject().get("children").getAsJsonArray()) {
                getCountersWithURLs(child_block, path, output);
            }
        } else {
            output.add(path);
        }
    }

    private static String parseUrl(String urlString) {
        if (Objects.equals(urlString, "")) {
            return "";
        }
        try {
            URL url = new URL(urlString);
            return url.getHost() + url.getPath();
        } catch (MalformedURLException e) {
            return "";
        }
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {

        List<String> params = Splitter.on("@@").splitToList(line);

        Map<String, String> kvParams = new HashMap<>();
        for (String param : params) {
            int splitIndex = param.indexOf('=');
            if (splitIndex > 0) {
                kvParams.put(param.substring(0, splitIndex), param.substring(splitIndex + 1));
            }
        }

        String dtype = ParseUtils.deNull(kvParams.get("dtype"));

        if (!Objects.equals(dtype, "clck")) {
            return;
        }

        Date clickDate;
        if (kvParams.containsKey("timestamp")) {
            clickDate = ParseUtils.parseDateInSeconds(kvParams.get("timestamp"));
        } else {
            clickDate = ParseUtils.parseDateInSeconds(params.get((params.size() - 3)));
        }
        UnsignedLong yandexUid = ParseUtils.parseUnsignedLong(kvParams.get("yandexuid"));
        UnsignedLong icookie = ParseUtils.parseUnsignedLong(kvParams.get("icookie"));
        String referer = kvParams.get("HTTP_REFERER");
        Integer timefrs = ParseUtils.parseInt(kvParams.get("timefrs"), 0);
        Integer timefs = ParseUtils.parseInt(kvParams.get("timefs"), 0);

        List<String> varsList = Splitter.on(",").splitToList(ParseUtils.deNull(kvParams.get("vars")));
        Map<String, String> vars = new HashMap<>();
        for (String var : varsList) {
            int splitIndex = var.indexOf('=');
            if (splitIndex > 0) {
                vars.put(var.substring(0, splitIndex), var.substring(splitIndex + 1));
            }
        }
        // Информация по браузеру - 1042=iemobile_11.0_trident_7.0_windowsphone_8.1_1_0_0
        List<String> browserOptions = Splitter.on("_").splitToList(ParseUtils.deNull(vars.get("1042")));
        boolean hasBrowserOptions = browserOptions.size() > 1;
        String uaBrowserEngine = hasBrowserOptions ? browserOptions.get(0) : "";
        String uaBrowserEngineVersion = hasBrowserOptions ? browserOptions.get(1) : "";
        String uaBrowserName = hasBrowserOptions ? browserOptions.get(2) : "";
        String uaBrowserVersion = hasBrowserOptions ? browserOptions.get(3) : "";
        String uaOsfamily = hasBrowserOptions ? browserOptions.get(4) : "";
        String uaOsVersion = hasBrowserOptions ? browserOptions.get(5) : "";
        Integer uaIsMobile = hasBrowserOptions ? ParseUtils.parseInt(browserOptions.get(6), 0) : 0;
        Integer uaIsTablet = hasBrowserOptions ? ParseUtils.parseInt(browserOptions.get(7), 0) : 0;
        Integer uaIsTv = hasBrowserOptions ? ParseUtils.parseInt(browserOptions.get(8), 0) : 0;
        // Эксперимент, списком test-id'ов - 1964=58905_58312_60002_54056_60078_59485",
        Integer[] testids;
        if (vars.get("1964") != null && !Objects.equals(vars.get("1964"), "")) {
            String[] parts = ParseUtils.deNull(vars.get("1964")).split("_");
            testids = new Integer[parts.length];
            for (int i = 0; i < parts.length; i++) {
                testids[i] = ParseUtils.parseInt(parts[i], 0);
            }
        } else {
            testids = new Integer[0];
        }

        // фолбэк на yandexuid в последнем поле лога
        if (yandexUid.equals(UnsignedLong.valueOf(0L))) {
            if (!ParseUtils.parseUnsignedLong(params.get(params.size() - 1), UnsignedLong.valueOf(0L)).equals(UnsignedLong.valueOf(0L))) {
                yandexUid = ParseUtils.parseUnsignedLong(params.get(params.size() - 1), UnsignedLong.valueOf(0L));
            }
        }

        if (kvParams.containsKey("lid") && kvParams.containsKey("sid")) {
            // Клики!
            String url = kvParams.get("url");
            Integer at = ParseUtils.parseInt(kvParams.get("at"), -1);
            String reqid = kvParams.get("sid");
            String path = kvParams.get("lid");
            String event_type = "click";
            String host = parseUrl(ParseUtils.deNull(kvParams.get("url")));

            context.write(clickDate, at, event_type, host, path, referer, reqid, url, yandexUid, icookie, timefrs,
                timefs, testids,
                uaBrowserEngine, uaBrowserEngineVersion, uaBrowserName, uaBrowserVersion,
                uaOsfamily, uaOsVersion, uaIsMobile, uaIsTablet, uaIsTv);
        } else if (kvParams.containsKey("events") && kvParams.containsKey("session_id")) {
            // Другие события!
            Integer at = 2;
            String reqid = kvParams.get("session_id");

            JsonElement eventsJson = new JsonParser().parse(URLDecoder.decode(kvParams.get("events")));
            JsonArray eventsArr = eventsJson.getAsJsonArray();

            for (JsonElement eventObject : eventsArr) {
                String parent = eventObject.getAsJsonObject().get("parent-path").getAsString();
                String event_type = eventObject.getAsJsonObject().get("event").getAsString();
                JsonElement url_element = eventObject.getAsJsonObject().get("url");  // optional
                String url = "";
                if (url_element != null) {
                    url = url_element.getAsString();
                }
                String host = parseUrl(ParseUtils.deNull(url));

                List<String> output = new ArrayList<>();
                for (JsonElement block : eventObject.getAsJsonObject().get("blocks").getAsJsonArray()) {
                    getCountersWithURLs(block, parent, output);
                }

                for (String path : output) {
                    context.write(clickDate, at, event_type, host, path, referer, reqid, url, yandexUid, icookie,
                        timefrs, timefs, testids,
                        uaBrowserEngine, uaBrowserEngineVersion, uaBrowserName, uaBrowserVersion,
                        uaOsfamily, uaOsVersion, uaIsMobile, uaIsTablet, uaIsTv
                    );
                }
            }
        }

    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
