package ru.yandex.market.logshatter;

import com.google.common.annotations.VisibleForTesting;
import org.apache.commons.lang.ArrayUtils;
import ru.yandex.clickhouse.settings.ClickHouseProperties;
import ru.yandex.clickhouse.util.ClickHouseRowBinaryStream;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 14/02/15
 */
public class LogBatch {
    private static final long MAX_TIME_IN_FUTURE_MILLIS = TimeUnit.DAYS.toMillis(365);
    private static final long MAX_TIME_IN_PAST_MILLIS = TimeUnit.DAYS.toMillis(20 * 365);

    private static final LocalDate EPOCH = LocalDate.ofEpochDay(0);

    private final long creationTimeMillis;

    private volatile Stream<String> linesStream;

    private final long dataOffset;
    private final long batchSizeBytes;

    private final long fileOffset;

    private final Duration readDuration;

    private final List<Column> columns;
    private final String sourceName;

    private volatile Duration parseDuration;
    private volatile int linesCount;
    private volatile int parseErrors;

    private final List<Date> parsedDates = new ArrayList<>();

    @VisibleForTesting
    volatile List<ParsingInProgressColumn> parsingInProgressColumns;
    private volatile byte[][] parsedColumns;

    private final CompletableFuture<Void> endOfProcessingFuture = new CompletableFuture<>();

    public LogBatch(Stream<String> linesStream, long dataOffset, long fileOffset, long batchSizeBytes,
                    Duration readDuration,
                    List<Column> columns, String sourceName) {
        this.creationTimeMillis = System.currentTimeMillis();
        this.linesStream = linesStream;
        this.batchSizeBytes = batchSizeBytes;
        this.dataOffset = dataOffset;
        this.readDuration = readDuration;
        this.fileOffset = fileOffset;
        this.columns = columns;
        this.sourceName = sourceName;
        this.parsingInProgressColumns = columns.stream()
            .skip(2)
            .map(column ->
                column.getType().isArray()
                    ? new ParsingInProgressArrayColumn(column)
                    : new ParsingInProgressRegularColumn(column)
            )
            .collect(Collectors.toList());
    }

    public void onParseComplete(Duration parseDuration, int linesCount, int parseErrors) {
        this.parseDuration = parseDuration;
        this.linesCount = linesCount;
        this.parseErrors = parseErrors;

        // Освобождаем память, которую занимали нераспаршенные данные
        this.linesStream = null;

        parsedColumns = parsingInProgressColumns.stream()
            .map(ParsingInProgressColumn::getParsedColumn)
            .toArray(byte[][]::new);

        parsingInProgressColumns = null;
    }

    public void onProcessingComplete() {
        endOfProcessingFuture.complete(null);
    }

    public void write(Date date, Object... fields) {
        validateData(date, fields);

        parsedDates.add(date);

        for (int i = 0; i < fields.length; i++) {
            parsingInProgressColumns.get(i).addValue(fields[i]);
        }
    }

    private void validateData(Date date, Object... fields) {
        if (!ColumnType.Date.validate(date)) {
            throw new IllegalArgumentException("Invalid date: " + date);
        }
        checkTimeDifference(date);
        if (columns.size() != fields.length + 2) {
            throw new IllegalArgumentException("Fields and columns sizes don't correspond");
        }
        for (int i = 0; i < fields.length; i++) {
            // i + 2 - т.к. первые две колонки это дата и таймстемп
            if (!columns.get(i + 2).getType().validate(fields[i])) {
                throw new IllegalArgumentException(
                    "Field #" + i + "(" + fields[i] + ") is not a " + columns.get(i + 2).getType()
                );
            }
        }
    }

    private static void checkTimeDifference(Date date) {
        long tsMillis = System.currentTimeMillis();
        if (date.getTime() - tsMillis > MAX_TIME_IN_FUTURE_MILLIS) {
            throw new IllegalArgumentException(
                "Date is more then " + TimeUnit.MILLISECONDS.toDays(MAX_TIME_IN_FUTURE_MILLIS) +
                    " days in future: " + date
            );
        }
        if (tsMillis - date.getTime() > MAX_TIME_IN_PAST_MILLIS) {
            throw new IllegalArgumentException(
                "Date is more then " + TimeUnit.MILLISECONDS.toDays(MAX_TIME_IN_PAST_MILLIS) +
                    " days in past: " + date
            );
        }
    }

    void writeTo(ClickHouseRowBinaryStream stream) throws IOException {
        //Native format
        stream.writeUnsignedLeb128(columns.size()); //Columns number
        stream.writeUnsignedLeb128(parsedDates.size()); //Rows number

        writeColumnHeader(stream, columns.get(0));
        for (Date date : parsedDates) {
            stream.writeUInt16(getUnsignedDaysSinceEpoch(date));
        }

        writeColumnHeader(stream, columns.get(1));
        for (Date date : parsedDates) {
            stream.writeDateTime(date);
        }

        for (int i = 0; i < parsedColumns.length; i++) {
            writeColumnHeader(stream, columns.get(i + 2));
            stream.writeBytes(parsedColumns[i]);
        }
    }

    private static void writeColumnHeader(ClickHouseRowBinaryStream stream, Column column) throws IOException {
        stream.writeString(column.getName());
        stream.writeString(column.getType().toClickhouseDDL());
    }

    private static short getUnsignedDaysSinceEpoch(Date date) {
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        short days = (short) ChronoUnit.DAYS.between(EPOCH, localDate);
        return (short) Short.toUnsignedInt(days);
    }

    public long getCreationTimeMillis() {
        return creationTimeMillis;
    }

    public long getFileOffset() {
        return fileOffset;
    }

    public List<Date> getParsedDates() {
        return parsedDates;
    }

    public long getBatchSizeBytes() {
        return batchSizeBytes;
    }

    public Stream<String> getLinesStream() {
        return linesStream;
    }

    public int getOutputSize() {
        return parsedDates.size();
    }

    public long getDataOffset() {
        return dataOffset;
    }

    public Duration getReadDuration() {
        return readDuration;
    }

    public Duration getParseDuration() {
        return parseDuration;
    }

    public int getParseErrors() {
        return parseErrors;
    }

    public int getLinesCount() {
        return linesCount;
    }

    public List<Column> getColumns() {
        return columns;
    }

    public String getSourceName() {
        return sourceName;
    }

    public CompletableFuture<Void> getEndOfProcessingFuture() {
        return endOfProcessingFuture;
    }


    @VisibleForTesting
    interface ParsingInProgressColumn {
        void addValue(Object value);
        byte[] getParsedColumn();
    }

    private static class ParsingInProgressRegularColumn implements ParsingInProgressColumn {
        private final Column column;
        private final InMemoryClickHouseRowBinaryStream valuesStream;

        private ParsingInProgressRegularColumn(Column column) {
            this.column = column;
            this.valuesStream = new InMemoryClickHouseRowBinaryStream();
        }

        @Override
        public void addValue(Object value) {
            try {
                column.getType().writeTo(value, valuesStream.stream);
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        @Override
        public byte[] getParsedColumn() {
            return valuesStream.toByteArray();
        }
    }

    private static class ParsingInProgressArrayColumn implements ParsingInProgressColumn {
        private final Column column;
        private final InMemoryClickHouseRowBinaryStream offsetsStream;
        private final InMemoryClickHouseRowBinaryStream elementsStream;
        private long offset = 0;

        private ParsingInProgressArrayColumn(Column column) {
            this.column = column;
            this.offsetsStream = new InMemoryClickHouseRowBinaryStream();
            this.elementsStream = new InMemoryClickHouseRowBinaryStream();
        }

        @Override
        public void addValue(Object value) {
            try {
                if (value instanceof Object[]) {
                    offset += ((Object[]) value).length;
                } else {
                    offset += ((Collection<?>) value).size();
                }
                offsetsStream.stream.writeUInt64(offset);
                column.getType().writeTo(value, elementsStream.stream);
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }

        @Override
        public byte[] getParsedColumn() {
            return ArrayUtils.addAll(offsetsStream.toByteArray(), elementsStream.toByteArray());
        }
    }

    private static class InMemoryClickHouseRowBinaryStream {
        final ClickHouseRowBinaryStream stream;
        private final ByteArrayOutputStream byteArrayOutputStream;

        private InMemoryClickHouseRowBinaryStream() {
            this.byteArrayOutputStream = new ByteArrayOutputStream();
            this.stream = new ClickHouseRowBinaryStream(
                byteArrayOutputStream,
                TimeZone.getDefault(),
                new ClickHouseProperties()
            );
        }

        byte[] toByteArray() {
            return byteArrayOutputStream.toByteArray();
        }
    }
}
