package ru.yandex.market.logshatter;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.clickhouse.ClickHouseConnection;
import ru.yandex.clickhouse.ClickHouseStatement;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.common.util.concurrent.ThreadFactories;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.health.HealthMetaDao;
import ru.yandex.market.health.KeyValueLog;
import ru.yandex.market.health.OutputInfo;
import ru.yandex.market.logshatter.config.ConfigurationService;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.meta.LogshatterMetaDao;
import ru.yandex.market.logshatter.output.ConfigOutputQueue;
import ru.yandex.market.logshatter.output.OutputQueue;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseQueue;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserContextImpl;
import ru.yandex.market.logshatter.parser.internal.LogshatterPerformanceLog;
import ru.yandex.market.logshatter.reader.ReadSemaphore;
import ru.yandex.market.logshatter.reader.SourceContext;
import ru.yandex.market.logshatter.rotation.DataRotationService;
import ru.yandex.market.logshatter.url.PageMatcher;
import ru.yandex.market.logshatter.useragent.UserAgentDetector;

import java.io.UncheckedIOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;
import java.util.zip.ZipException;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 14/02/15
 */
public class LogShatterService implements InitializingBean, Runnable, Thread.UncaughtExceptionHandler {
    private static final Logger log = LogManager.getLogger();
    private static final Logger performanceLog = LogManager.getLogger("performance");
    private static final Logger parserLog = LogManager.getLogger("parser");

    private static final long MAX_TIME_DIFFERENCE_MILLIS = TimeUnit.SECONDS.toMillis(5);

    private final ParseQueue parseQueue = new ParseQueue();
    private final OutputQueue outputQueue = new OutputQueue();
    private final AtomicLong lastOutputErrorMillis = new AtomicLong();
    private ClickHouseConnection clickHouseConnection;
    private LogshatterMetaDao logshatterMetaDao;
    private HealthMetaDao healthMetaDao;
    private PageMatcher pageMatcher;
    private LogShatterMonitoring monitoring;
    private ReadSemaphore readSemaphore;
    private DataRotationService dataRotationService;

    private int parseThreadCount = 2;
    private int minOutputThreadCount = 1;
    private int maxOutputThreadCount = 3;
    private Map<String, Float> dataSampling = Collections.emptyMap();
    private int outputBatchSize = 50_000;
    private int outputErrorsCountThreshold = 10;
    private int outputIdleTimeSeconds = 0;
    private int sleepBetweenErrorsSeconds = 5;
    private int outputInfoRoundSeconds = 60;
    private int logLineLengthWarnLimit = 500_000;
    private int logLineLengthErrorLimit = 1_000_000;
    private volatile boolean running = true;
    private boolean skipObsoleteData = false;
    private int httpPort = 32184;
    private ExecutorService parseExecutorService;
    private ExecutorService outputExecutorService;
    private ConfigurationService configurationService;
    private boolean writeOutputMeta = true;

    @Override
    public void afterPropertiesSet() throws Exception {
        Map<String, Float> remapped = new HashMap<>(dataSampling.size());
        dataSampling.forEach((key, value) -> {
            if (!key.contains(".")) {
                key = configurationService.getDefaultClickHouseDatabase() + "." + key;
            }

            remapped.put(key, value);
        });

        dataSampling = remapped;

        checkTimeDifference();

        init();
        Thread thread = new Thread(this, "LogShatterService");
        thread.setDaemon(false);
        thread.start();
    }

    /**
     * Проверяем, что в java и в Clickhouse одинаковое время, не больше чем MAX_TIME_DIFFERENCE_MILLIS,
     * а самое главное - одинаковый часовой пояс. Иначе начнеться кровькишкарасчленёнка
     */
    private void checkTimeDifference() {
        try (ClickHouseStatement clickHouseStatement = clickHouseConnection.createStatement()) {
            try (ResultSet resultSet = clickHouseStatement.executeQuery("SELECT toUnixTimestamp(now())")) {
                resultSet.next();
                Date clickHouseDate = new Date(TimeUnit.SECONDS.toMillis(resultSet.getLong(1)));
                Date javaDate = new Date();
                long differenceMillis = Math.abs(clickHouseDate.getTime() - javaDate.getTime());
                log.info("ClickHouse and local clock difference: {}ms", differenceMillis);
                if (differenceMillis > MAX_TIME_DIFFERENCE_MILLIS) {
                    throw new RuntimeException(
                        "Difference between java time and clickhouse time is more than " +
                            MAX_TIME_DIFFERENCE_MILLIS + "ms. Maybe problem with timezones. " +
                            "Java time " + javaDate + ", clickhouse time: " + clickHouseDate
                    );
                }
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private void init() {
        Thread.setDefaultUncaughtExceptionHandler(this);
        initHook();
        initThreadPools();
    }

    @Override
    public void uncaughtException(Thread t, Throwable e) {
        log.fatal("Fatal unhandled error in thread " + t.getName() + ". Terminating", e);
        e.printStackTrace();
        running = false;
        System.exit(1);
    }

    private void initThreadPools() {
        parseExecutorService = createExecutorService(parseThreadCount, "parser-");
        for (int i = 0; i < parseThreadCount; i++) {
            parseExecutorService.submit(new ParserWorker());
        }

        outputExecutorService = createExecutorService(maxOutputThreadCount, "output-");
        for (int i = 1; i <= maxOutputThreadCount; i++) {
            outputExecutorService.submit(new OutputWorker(i, outputIdleTimeSeconds));
        }
    }

    public ExecutorService createExecutorService(final int threadCount, String prefix) {
        return new ThreadPoolExecutor(
            threadCount, threadCount, 0L, TimeUnit.MILLISECONDS,
            new LinkedBlockingQueue<Runnable>(), ThreadFactories.named(prefix)
        ) {
            @Override
            protected void afterExecute(Runnable r, Throwable t) {
                if (t != null) {
                    uncaughtException(Thread.currentThread(), t);
                }
                if (r instanceof Future) {
                    Future future = (Future) r;
                    try {
                        future.get();
                    } catch (InterruptedException ignored) {
                    } catch (ExecutionException e) {
                        uncaughtException(Thread.currentThread(), e);
                    }
                }
            }
        };
    }

    private void initHook() {
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable() {
            @Override
            public void run() {
                running = false;
                parseExecutorService.shutdownNow();
                outputExecutorService.shutdownNow();
                for (int i = 0; i < 10; i++) {
                    if (outputExecutorService.isTerminated()) {
                        break;
                    }
                    try {
                        log.info("Waiting for termination...");
                        outputExecutorService.awaitTermination(1, TimeUnit.SECONDS);
                    } catch (InterruptedException e) {
                        log.error("Shutdown hook interrupted", e);
                    }
                }
            }
        }));
    }

    @Override
    public void run() {
        while (!Thread.interrupted() && running) {
            logStat();
            try {
                TimeUnit.MINUTES.sleep(1);
            } catch (InterruptedException ignored) {
            }
        }
        log.info("Finishing");
    }

    private void logStat() {
        int openFiles = readSemaphore.getOpenFilesCount();
        log.info(
            "Internal queue size: " + LogShatterUtil.bytesToMb(readSemaphore.getQueueSizeBytes()) + "mb" +
                ". OpenFiles: " + openFiles +
                ". ParseQueue:" + parseQueue.toString() +
                ". OutputQueue:" + outputQueue.toString()
        );
        KeyValueLog.log("internalQueueSizeMb", LogShatterUtil.bytesToMb(readSemaphore.getQueueSizeBytes()));
        KeyValueLog.log("internalQueueSizePercent", readSemaphore.getInternalQueueUsagePercent());
        KeyValueLog.log("openFiles", openFiles);
        KeyValueLog.log("outputThreadCount", getCurrentOutputThreadCount());
    }

    public void addToParseQueue(SourceContext sourceContext) {
        parseQueue.add(sourceContext);
    }

    public boolean isRunning() {
        return running;
    }

    public Map<String, Float> getDataSampling() {
        return dataSampling;
    }

    public void setDataSampling(Map<String, Float> dataSampling) {
        this.dataSampling = dataSampling;
    }

    private boolean canOutput(int threadNum) {
        if (threadNum <= minOutputThreadCount) {
            return true;
        }
        return threadNum <= getCurrentOutputThreadCount();
    }

    private int getCurrentOutputThreadCount() {
        int workersCount = (int) Math.round(readSemaphore.getInternalQueueUsagePercent() / 100 * maxOutputThreadCount);
        return Math.max(workersCount, minOutputThreadCount);
    }

    private List<String> getColumnNames(LogBatch logBatch) {
        List<String> columnNames = new ArrayList<>();
        for (Column column : logBatch.getColumns()) {
            columnNames.add(column.getName());
        }
        return columnNames;
    }

    static class SaveToClickhouseResult {
        private final long outputStartTimeMillis;
        private final long outputEndTimeMillis;
        private final int tryNumber;

        SaveToClickhouseResult(long outputStartTimeMillis, long outputEndTimeMillis, int tryNumber) {
            this.outputStartTimeMillis = outputStartTimeMillis;
            this.outputEndTimeMillis = outputEndTimeMillis;
            this.tryNumber = tryNumber;
        }

        int getOutputTimeMillis() {
            return (int) (outputEndTimeMillis - outputStartTimeMillis);
        }
    }

    private Optional<SaveToClickhouseResult> saveToClickhouse(MultiMap<SourceContext, LogBatch> sourceContextToLogBatchesMap, LogShatterConfig config) throws InterruptedException {
        int tryNumber = 1;

        while (true) {
            List<LogBatch> logBatches = sourceContextToLogBatchesMap.entrySet().stream()
                .filter(entry -> !entry.getKey().isFinished())
                .flatMap(entry -> entry.getValue().stream())
                .collect(Collectors.toList());

            int totalOutputSize = logBatches.stream().mapToInt(LogBatch::getOutputSize).sum();
            if (totalOutputSize == 0) {
                return Optional.empty();
            }

            long outputStartTimeMillis = System.currentTimeMillis();

            try {
                List<String> columns = getColumnNames(logBatches.get(0));
                String sql = "INSERT INTO " + config.getInsertTableName() + " "
                    + columns.stream().collect(Collectors.joining(", ", "(", ")"));

                saveToClickhouseInNativeFormat(sql, logBatches);

                long outputEndTimeMillis = System.currentTimeMillis();
                return Optional.of(new SaveToClickhouseResult(outputStartTimeMillis, outputEndTimeMillis, tryNumber));
            } catch (Exception e) {
                long currentTimeMillis = System.currentTimeMillis();
                writePerformanceLog(
                    new Date(currentTimeMillis), LogshatterPerformanceLog.OutputStatus.FAILURE, config,
                    logBatches, (int) (currentTimeMillis - outputStartTimeMillis), -1, tryNumber
                );

                log.error(
                    "Failed to save batch. Waiting " + sleepBetweenErrorsSeconds + " seconds. " +
                        "Table: " + config.getInsertTableName(), e
                );
                lastOutputErrorMillis.set(System.currentTimeMillis());
                if (tryNumber >= outputErrorsCountThreshold) {
                    monitoring.getHostCritical().addTemporaryCritical(
                        "Output", "Failed to save to clickhouse",
                        1, TimeUnit.MINUTES
                    );
                }
                TimeUnit.SECONDS.sleep(sleepBetweenErrorsSeconds);
                tryNumber++;
            }
        }
    }

    private void saveToClickhouseInNativeFormat(String sql, List<LogBatch> logBatches) throws SQLException {
        ClickHouseStatement statement = clickHouseConnection.createStatement();

        statement.sendNativeStream(sql, stream -> {
            for (LogBatch logBatch : logBatches) {
                logBatch.writeTo(stream);
            }
        });
    }

    private void outputInfo(String table, List<LogBatch> logBatches) {
        RangeSet<Integer> rangeSet = TreeRangeSet.create();
        int count = 0;
        for (LogBatch batch : logBatches) {

            if (batch.getOutputSize() > 0) {
                count += batch.getOutputSize();
                for (Date date : batch.getParsedDates()) {
                    int timeSeconds = (int) TimeUnit.MILLISECONDS.toSeconds(date.getTime());
                    if (!rangeSet.contains(timeSeconds)) {
                        rangeSet.add(LogShatterUtil.createRange(timeSeconds, outputInfoRoundSeconds));
                    }
                }
            }
        }

        if (count > 0) {
            Set<Range<Integer>> ranges = rangeSet.asRanges();
            List<OutputInfo> outputInfos = new ArrayList<>(ranges.size());
            for (Range<Integer> range : ranges) {
                outputInfos.add(new OutputInfo(table, range.lowerEndpoint(), range.upperEndpoint(), count));
            }
            saveOutputInfo(outputInfos);
        }
    }

    private void saveOutputInfo(List<OutputInfo> outputInfos) {
        if (!writeOutputMeta){
            return;
        }
        int tryNumber = 1;
        while (!Thread.interrupted()) {
            try {
                healthMetaDao.save(outputInfos);
                return;
            } catch (Exception e) {
                log.error("Failed to save output stat", e);
                if (tryNumber >= outputErrorsCountThreshold) {
                    monitoring.getHostCritical().addTemporaryCritical(
                        "MongoDb", "Failed to save output stat",
                        1, TimeUnit.MINUTES
                    );
                }
                try {
                    TimeUnit.SECONDS.sleep(sleepBetweenErrorsSeconds);
                } catch (InterruptedException ignored) {
                }
                tryNumber++;
            }
        }
    }

    private void completeBatch(MultiMap<SourceContext, LogBatch> sourceLogBatches) throws InterruptedException {
        for (Map.Entry<SourceContext, List<LogBatch>> entry : sourceLogBatches.entrySet()) {
            updateOffsets(entry.getKey(), entry.getValue());
        }
        saveMeta(sourceLogBatches.keySet());
        decrementQueue(sourceLogBatches);
        for (List<LogBatch> logBatches : sourceLogBatches.values()) {
            for (LogBatch logBatch : logBatches) {
                logBatch.onProcessingComplete();
            }
        }
    }

    private void saveMeta(Collection<SourceContext> sourceContexts) throws InterruptedException {
        int tryNumber = 1;
        while (!Thread.interrupted()) {
            try {
                List<SourceContext> notFinishedSourceContexts = sourceContexts.stream()
                    .filter(sourceContext -> !sourceContext.isFinished())
                    .collect(Collectors.toList());
                if (!notFinishedSourceContexts.isEmpty()) {
                    logshatterMetaDao.save(notFinishedSourceContexts);
                }
                return;
            } catch (Exception e) {
                log.error("Failed to save metadata", e);
                if (tryNumber >= outputErrorsCountThreshold) {
                    monitoring.getHostCritical().addTemporaryCritical(
                        "MetaDataSave", "Failed to save meta",
                        1, TimeUnit.MINUTES
                    );
                }
                TimeUnit.SECONDS.sleep(sleepBetweenErrorsSeconds);
                tryNumber++;
            }
        }
    }

    private void updateOffsets(SourceContext sourceContext, List<LogBatch> batches) {
        for (LogBatch batch : batches) {
            if (batch.getFileOffset() >= 0) {
                sourceContext.setFileOffset(batch.getFileOffset());
            }
            if (batch.getDataOffset() >= 0) {
                sourceContext.setDataOffset(batch.getDataOffset());
            }
        }
    }

    private void decrementQueue(MultiMap<SourceContext, LogBatch> sourceLogBatches) {
        long processedBytes = 0;
        for (Map.Entry<SourceContext, List<LogBatch>> entry : sourceLogBatches.entrySet()) {
            final ReadSemaphore.QueuesCounter queuesCounter = entry.getKey().getQueuesCounter();

            for (LogBatch logBatch : entry.getValue()) {
                processedBytes += logBatch.getBatchSizeBytes();
                queuesCounter.decrement(logBatch.getBatchSizeBytes());
            }
        }

        readSemaphore.decrementGlobalQueue(processedBytes);
    }

    private void writePerformanceLog(Date date, LogshatterPerformanceLog.OutputStatus status,
                                     LogShatterConfig logShatterConfig, List<LogBatch> batches,
                                     int outputTimeMillis, int fullOutputTimeMillis, int tryNumber) {
        performanceLog.info(
            LogshatterPerformanceLog.format(
                date, status, logShatterConfig, batches, outputTimeMillis, fullOutputTimeMillis, tryNumber
            )
        );
    }

    @Required
    public void setHealthMetaDao(HealthMetaDao healthMetaDao) {
        this.healthMetaDao = healthMetaDao;
    }

    @Required
    public void setClickHouseConnection(ClickHouseConnection clickHouseConnection) {
        this.clickHouseConnection = clickHouseConnection;
    }

    @Required
    public void setLogshatterMetaDao(LogshatterMetaDao logshatterMetaDao) {
        this.logshatterMetaDao = logshatterMetaDao;
    }

    @Required
    public void setMonitoring(LogShatterMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    @Required
    public void setReadSemaphore(ReadSemaphore readSemaphore) {
        this.readSemaphore = readSemaphore;
    }

    @Required
    public void setPageMatcher(PageMatcher pageMatcher) {
        this.pageMatcher = pageMatcher;
    }

    public void setParseThreadCount(int parseThreadCount) {
        this.parseThreadCount = parseThreadCount;
    }

    public void setMinOutputThreadCount(int minOutputThreadCount) {
        this.minOutputThreadCount = minOutputThreadCount;
    }

    public void setMaxOutputThreadCount(int maxOutputThreadCount) {
        this.maxOutputThreadCount = maxOutputThreadCount;
    }

    public void setOutputBatchSize(int outputBatchSize) {
        this.outputBatchSize = outputBatchSize;
    }

    public void setOutputIdleTimeSeconds(int outputIdleTimeSeconds) {
        this.outputIdleTimeSeconds = outputIdleTimeSeconds;
    }

    public void setLogLineLengthWarnLimit(int logLineLengthWarnLimit) {
        this.logLineLengthWarnLimit = logLineLengthWarnLimit;
    }

    public void setLogLineLengthErrorLimit(int logLineLengthErrorLimit) {
        this.logLineLengthErrorLimit = logLineLengthErrorLimit;
    }

    public void setDataRotationService(DataRotationService dataRotationService) {
        this.dataRotationService = dataRotationService;
    }

    public void setSkipObsoleteData(boolean skipObsoleteData) {
        this.skipObsoleteData = skipObsoleteData;
    }

    public void setConfigurationService(ConfigurationService configurationService) {
        this.configurationService = configurationService;
    }

    public void setHttpPort(int httpPort) {
        this.httpPort = httpPort;
    }

    public void setWriteOutputMeta(boolean writeOutputMeta) {
        this.writeOutputMeta = writeOutputMeta;
    }

    @VisibleForTesting
    public class ParserWorker implements Runnable {
        private static final int LINES_BATCH_SIZE = 1000;

        @Override
        public void run() {
            while (!Thread.interrupted() && running) {
                try {
                    parseOneSourceContext();
                } catch (InterruptedException ignored) {
                }
            }

            log.info("Parser thread finished");
        }

        @VisibleForTesting
        public void parseOneSourceContext() throws InterruptedException {
            SourceContext sourceContext = parseQueue.takeLock();

            LogBatch logBatch;
            while ((logBatch = sourceContext.getParseQueue().poll()) != null) {
                if (sourceContext.isFinished()) {
                    sourceContext.getQueuesCounter().decrement(logBatch.getBatchSizeBytes());
                    readSemaphore.decrementGlobalQueue(logBatch.getBatchSizeBytes());

                    // sourceContext.isFinished() == true означает что мы в процессе остановки чтения данных по этому
                    // SourceContext'у (например логброкерная сессия закрывается), и нужно выбросить все данные, которые
                    // мы уже прочитали, но не записали в Кликхаус.
                    // Не вызываем logBatch.onProcessingComplete(), потому что не хотим чтобы закоммитились оффсеты.
                    // Хотим чтобы Логброкер прислал эти же данные ещё раз в следующий раз.

                    log.info(
                        "Source context {} is finished, not parsing. Skipping batch: file offset = {}, " +
                            "data offset = {}, data size compressed = {}",
                        sourceContext, logBatch.getFileOffset(), logBatch.getDataOffset(), logBatch.getBatchSizeBytes()
                    );
                } else {
                    parseBatch(sourceContext, logBatch);
                }
            }

            parseQueue.returnLock(sourceContext);
        }

        @VisibleForTesting
        void parseBatch(SourceContext sourceContext, LogBatch logBatch) {
            Stopwatch stopwatch = Stopwatch.createStarted();

            float sampleRatio = dataSampling.getOrDefault(
                sourceContext.getLogShatterConfig().getTableName(), 1.0f);

            AtomicInteger linesCount = new AtomicInteger();
            AtomicInteger errorCount = new AtomicInteger();

            if (sampleRatio != -1) {
                parseBatchLines(sourceContext, logBatch, sampleRatio, linesCount, errorCount);
            } else {
                log.debug("Sample ratio equals -1 for table {}. Skipping batch parsing: file offset = {}, " +
                        "data offset = {}, data size compressed = {}",
                    sourceContext.getLogShatterConfig().getTableName(), logBatch.getFileOffset(),
                    logBatch.getDataOffset(), logBatch.getBatchSizeBytes()
                );
            }

            logBatch.onParseComplete(stopwatch.elapsed(), linesCount.intValue(), errorCount.intValue());

            sourceContext.getOutputQueue().add(logBatch);
            outputQueue.add(sourceContext);
        }

        private void parseBatchLines(SourceContext sourceContext, LogBatch logBatch, float sampleRatio,
                                     AtomicInteger linesCount, AtomicInteger errorCount) {

            UserAgentDetector userAgentDetector = configurationService.getUserAgentDetector();

            ParserContext parserContext = new ParserContextImpl(
                logBatch, sourceContext, pageMatcher, skipObsoleteData, sampleRatio, userAgentDetector);
            LogParser parser = sourceContext.getLogParser();

            try {
                logBatch.getLinesStream().forEach(
                        line -> {
                            linesCount.getAndIncrement();
                            try {
                                if (line.length() > logLineLengthErrorLimit) {
                                    parserLog.error(
                                            "Log line is " + line.length() + " symbols (>" + logLineLengthErrorLimit + "). Skipping!" +
                                                    " File " + sourceContext.getPath() + " from host " + sourceContext.getHost()
                                                    + ", line (fist 1000 symbols):\n" + line.substring(0, LINES_BATCH_SIZE)
                                    );
                                } else {
                                    if (line.length() > logLineLengthWarnLimit) {
                                        parserLog.warn(
                                                "Log line is " + line.length() + " symbols (>" + logLineLengthWarnLimit + "). " +
                                                        "File " + sourceContext.getPath() + " from host " + sourceContext.getHost()
                                                        + ", line (fist 1000 symbols):\n" + line.substring(0, LINES_BATCH_SIZE)
                                        );
                                    }
                                    parser.parse(line, parserContext);
                                }
                            } catch (Exception e) {
                                errorCount.getAndIncrement();
                                sourceContext.getErrorLogger().addError(e, line);
                            }
                        }
                );
            } catch (UncheckedIOException e) {
                if (e.getCause() != null && e.getCause() instanceof ZipException) {
                    log.error(
                            "Cannot unzip logBatch, skipping. Error of type {}: {} failed to parse file {} from host {}." +
                                    " SourceContext info: name: {}, lbTopic: {}, fileOffset: {}, dataOffset: {}," +
                                    " sourceKey: {}, instanceId: {}",
                            e.getClass().getName(),
                            parser.getClass().getName(),
                            sourceContext.getPath(),
                            sourceContext.getHost(),
                            sourceContext.getName(),
                            sourceContext.getLogBrokerTopic(),
                            sourceContext.getFileOffset(),
                            sourceContext.getDataOffset(),
                            sourceContext.getSourceKey(),
                            sourceContext.getInstanceId());
                    return;
                }
                throw e;
            }

            sourceContext.getErrorLogger().batchParsed();
            errorCount.intValue();
        }
    }

    @VisibleForTesting
    public class OutputWorker implements Runnable {

        private final int number;
        private final int idleTimeSeconds;

        public OutputWorker(int number, int idleTimeSeconds) {

            this.number = number;
            this.idleTimeSeconds = idleTimeSeconds;
        }

        @Override
        public void run() {
            while (!Thread.interrupted() && running) {
                try {
                    while (!canOutput(number)) {
                        TimeUnit.SECONDS.sleep(1);
                    }
                    outputOnce();
                } catch (InterruptedException ignored) {
                }
            }
            log.info("Output thread finished");
        }

        @VisibleForTesting
        public void outputOnce() throws InterruptedException {
            ConfigOutputQueue configOutputQueue = outputQueue.takeLock();
            MultiMap<SourceContext, LogBatch> fileLogBatches = new MultiMap<>();
            List<LogBatch> logBatches = new ArrayList<>();

            List<SourceContext> lockedSourceContexts = new ArrayList<>();

            int batchSize = 0;

            Stopwatch sw = Stopwatch.createStarted();
            try {
                while (batchSize < outputBatchSize && !configOutputQueue.isEmpty()) {
                    SourceContext sourceContext = configOutputQueue.takeLock();
                    lockedSourceContexts.add(sourceContext);

                    boolean canSave = sourceContext.beginSave();
                    LogBatch logBatch;
                    while (batchSize < outputBatchSize && (logBatch = sourceContext.getOutputQueue().poll()) != null) {
                        if (canSave) {
                            logBatches.add(logBatch);
                            fileLogBatches.append(sourceContext, logBatch);
                            batchSize += logBatch.getOutputSize();
                        } else {
                            sourceContext.getQueuesCounter().decrement(logBatch.getBatchSizeBytes());
                            readSemaphore.decrementGlobalQueue(logBatch.getBatchSizeBytes());

                            log.info(
                                "Source context {} is finished, skipping parsed batch: file offset = {}, " +
                                    "data offset = {}, parsed line count = {}",
                                sourceContext, logBatch.getFileOffset(), logBatch.getDataOffset(), logBatch.getOutputSize()
                            );
                        }
                    }
                }
            } finally {
                if (batchSize >= outputBatchSize) {
                    outputQueue.returnLockAndMaybeAddFirst(configOutputQueue);
                } else {
                    outputQueue.returnLockAndMaybeAddLast(configOutputQueue);
                }
            }

            try {
                if (!logBatches.isEmpty()) {
                    LogShatterConfig logShatterConfig = configOutputQueue.getLogShatterConfig();
                    Optional<SaveToClickhouseResult> saveResult = saveToClickhouse(fileLogBatches, logShatterConfig);

                    outputInfo(logShatterConfig.getTableName(), logBatches);
                    completeBatch(fileLogBatches);
                    long fullOutputEndTimeMillis = System.currentTimeMillis();

                    writePerformanceLog(
                        new Date(fullOutputEndTimeMillis),
                        LogshatterPerformanceLog.OutputStatus.SUCCESS, logShatterConfig, logBatches,
                        saveResult.map(SaveToClickhouseResult::getOutputTimeMillis).orElse(-1),
                        saveResult.map(r -> (int) (fullOutputEndTimeMillis - r.outputStartTimeMillis)).orElse(-1),
                        saveResult.map(r -> r.tryNumber).orElse(0)
                    );
                }
            } finally {
                for (SourceContext sourceContext : lockedSourceContexts) {
                    sourceContext.completeSave();
                    configOutputQueue.returnLock(sourceContext);
                }
            }

            outputQueue.returnLockAndMaybeAddLast(configOutputQueue);

            if (batchSize < outputBatchSize) {
                TimeUnit.SECONDS.sleep(outputIdleTimeSeconds - sw.elapsed().getSeconds());
            }
        }
    }
}
