package ru.yandex.market.logshatter;

import com.google.common.collect.SortedSetMultimap;
import com.google.common.collect.TreeMultimap;
import com.google.common.hash.Hasher;
import com.google.common.hash.Hashing;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.clickhouse.ddl.DdlQuery;
import ru.yandex.market.logshatter.config.ConfigurationService;
import ru.yandex.market.logshatter.config.ddl.ManualDDLExecutionResult;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLOnHostResult;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLService;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLWorker;
import ru.yandex.market.logshatter.rotation.DataRotationService;
import ru.yandex.market.monitoring.ComplicatedMonitoring;
import ru.yandex.market.monitoring.MonitoringStatus;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 18/10/2017
 */
@RestController
public class LogshatterRestController {

    private static final Gson GSON = new GsonBuilder().disableHtmlEscaping().setPrettyPrinting().create();
    public static final String DROP_PARTITIONS_METHOD = "/getDropPartitionsScripts";

    private final LogShatterMonitoring monitoring;
    private final ConfigurationService configurationService;
    private final UpdateDDLService updateDDLService;
    private final DataRotationService dataRotationService;
    private final int httpPort;

    public LogshatterRestController(LogShatterMonitoring monitoring,
                                    ConfigurationService configurationService,
                                    DataRotationService dataRotationService,
                                    UpdateDDLService updateDDLService,
                                    @Value("${logshatter.http.port}") int httpPort) {
        this.monitoring = monitoring;
        this.configurationService = configurationService;
        this.updateDDLService = updateDDLService;
        this.dataRotationService = dataRotationService;
        this.httpPort = httpPort;
    }

    @RequestMapping(value = "/ping", produces = "text/plain; charset=UTF-8")
    public ResponseEntity<String> ping() {
        return buildMonitoringResponse(monitoring.getOverallResult());
    }

    @RequestMapping(value = "/monitoringHostCritical", produces = "text/plain; charset=UTF-8")
    public ResponseEntity<String> monitoringHostCritical() {
        return buildMonitoringResponse(monitoring.getHostCritical().getResult());
    }

    @RequestMapping(value = "/monitoringClusterCritical", produces = "text/plain; charset=UTF-8")
    public ResponseEntity<String> monitoringClusterCritical() {
        return buildMonitoringResponse(monitoring.getClusterCritical().getResult());
    }

    private static ResponseEntity<String> buildMonitoringResponse(ComplicatedMonitoring.Result result) {
        HttpStatus status =
            result.getStatus() == MonitoringStatus.CRITICAL ? HttpStatus.INTERNAL_SERVER_ERROR : HttpStatus.OK;
        return ResponseEntity.status(status).body(result.toString());
    }

    @RequestMapping(value = "/getManualDDL", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public String getManualDDL() {
        UpdateDDLWorker ddlWorker = configurationService.getDDLWorker();
        List<DDL> ddlThatNeedsConfirmation = ddlWorker.getManualDDLs();
        ManualDdl manualDdl = toManualDDL(ddlThatNeedsConfirmation);
        return GSON.toJson(manualDdl);
    }

    @PostMapping(value = "/applyManualDDL", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public ResponseEntity<String> applyManualDDL(@RequestParam(value = "hash") String expectedHash) {
        UpdateDDLWorker ddlWorker = configurationService.getDDLWorker();
        List<DDL> ddlThatNeedsConfirmation = ddlWorker.getManualDDLs();

        ManualDdl manualDdl = toManualDDL(ddlThatNeedsConfirmation);
        if (!expectedHash.endsWith(manualDdl.hash)) {
            return ResponseEntity.badRequest().body("Invalid hash for manual DDL");
        }

        ManualDDLExecutionResult manualDDLExecutionResult = ddlWorker.executeManualDDLs(ddlThatNeedsConfirmation);
        return ResponseEntity.ok(GSON.toJson(manualDDLExecutionResult));
    }


    @RequestMapping(value = DROP_PARTITIONS_METHOD, produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public String getDropPartitionsScripts() {
        return GSON.toJson(dataRotationService.getObsoletePartitionsByHostMap().asMap());
    }

    @PostMapping(value = "/applyDDLOnHost", produces = MediaType.APPLICATION_JSON_UTF8_VALUE)
    public String applyDdlOnHost(@RequestParam String host, @RequestParam(defaultValue = "false") boolean applyManual) {
        UpdateDDLOnHostResult result = updateDDLService.applyDdlOnHost(
            configurationService.getConfigs(), host, applyManual
        );
        return GSON.toJson(result);
    }


    private ManualDdl toManualDDL(List<DDL> ddls) {
        Hasher hasher = Hashing.md5().newHasher();
        SortedSetMultimap<String, String> hostToDll = TreeMultimap.create();

        for (DDL ddl : ddls) {
            for (DdlQuery ddlQuery : ddl.getManualUpdates()) {
                hostToDll.put(ddl.getHost(), ddlQuery.getQueryString());
            }
            for (String error : ddl.getErrors()) {
                hostToDll.put(ddl.getHost(), "Error (cannot be fixed by logshatter): " + error);
            }
        }

        for (Map.Entry<String, String> hostToDdlEntry : hostToDll.entries()) {
            hasher.putString(hostToDdlEntry.getKey(), Charset.defaultCharset());
            hasher.putString(hostToDdlEntry.getValue(), Charset.defaultCharset());
        }

        String host;
        try {
            host = InetAddress.getLocalHost().getCanonicalHostName();
        } catch (UnknownHostException e) {
            throw new RuntimeException(e);
        }
        String hash = hasher.hash().toString();
        String url = String.format("http://%s:%d/applyManualDDL?hash=%s", host, httpPort, hash);

        return new ManualDdl(hash, hostToDll.asMap(), url);
    }

    private static class ManualDdl {
        private final Map<String, Collection<String>> ddl;
        private final String url;
        private final String hash;

        private ManualDdl(String hash, Map<String, Collection<String>> ddl, String url) {
            this.hash = hash;
            this.ddl = ddl;
            this.url = url;
        }

    }


}
