package ru.yandex.market.logshatter.config;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.fge.jackson.JacksonUtils;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import com.github.fge.jsonschema.core.util.AsJson;
import com.github.fge.jsonschema.main.JsonSchemaFactory;
import com.github.fge.jsonschema.main.JsonValidator;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Splitter;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryNTimes;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.market.clickhouse.ddl.ClickHouseDdlServiceOld;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinition;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinitionImpl;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnTypeUtils;
import ru.yandex.market.clickhouse.ddl.engine.DistributedEngine;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.LogShatterMonitoring;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLService;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLWorker;
import ru.yandex.market.logshatter.generic.json.JsonParserConfig;
import ru.yandex.market.logshatter.parser.LogParserProvider;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.useragent.UserAgentDetector;
import ru.yandex.market.monitoring.MonitoringUnit;

import javax.naming.ConfigurationException;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 24/02/15
 */
public class ConfigurationService implements InitializingBean {
    private static final String LOCAL_REPLICATED_TABLE_POSTFIX = "_lr";
    private static final String VALIDATION_SCHEMA_PATH = "LogShatterConfigSchema.json";
    private static final JsonValidator VALIDATOR = JsonSchemaFactory.byDefault().getValidator();
    private static final JsonNode VALIDATION_SCHEMA;
    private static final ObjectMapper NODE_READER;
    private static final Pattern TABLE_NAME_REXEXP = Pattern.compile(
        "^([a-zA-Z_][0-9a-zA-Z_]*\\.)?[a-zA-Z_][0-9a-zA-Z_]*$"
    );

    private static final Logger log = LogManager.getLogger();

    private static final JsonParser JSON_PARSER = new JsonParser();

    static {
        JsonFactory jsonFactory = new JsonFactory();
        jsonFactory.enable(com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_COMMENTS);
        NODE_READER = new ObjectMapper(jsonFactory);

        try {
            String schema = IOUtils.toString(ClassLoader.getSystemResourceAsStream(VALIDATION_SCHEMA_PATH));
            VALIDATION_SCHEMA = NODE_READER.readTree(new StringReader(schema));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private File configDir;
    private List<LogSource> defaultSources;
    private String defaultClickHouseDatabase;
    private ClickHouseDdlServiceOld clickhouseDdlService;
    private LogShatterMonitoring monitoring;

    private CuratorFramework curatorFramework;
    private String zookeeperQuorum;
    private int zookeeperTimeoutMillis = (int) TimeUnit.SECONDS.toMillis(30);
    private String logshatterZookeeperPrefix;
    private int defaultDataRotationDays;
    private List<LogShatterConfig> configs;
    private Multimap<LogSource, LogShatterConfig> sourceToConfigs;
    private UpdateDDLService updateDDLService;
    private UpdateDDLWorker ddlWorker;
    private int httpPort = 32184;
    private String clickHouseZookeeperTablePrefix;
    private UserAgentDetector userAgentDetector;


    public static ParserConfig getParserConfig(JsonObject configObject) {
        if (!configObject.has("parser")) {
            return null;
        }
        JsonObject parser = configObject.getAsJsonObject("parser");
        Gson gson = new Gson();
        ParserConfig parserConfig = gson.fromJson(parser, ParserConfig.class);

        if (parser.has("requirement")) {
            MultiMap<String, String> requirement = parserConfig.getRequiredMap();
            for (Map.Entry<String, JsonElement> entry : parser.getAsJsonObject("requirement").entrySet()) {
                JsonArray valuesArray = entry.getValue().getAsJsonArray();
                List<String> values = new ArrayList<>(valuesArray.size());
                for (int i = 0; i < valuesArray.size(); i++) {
                    values.add(valuesArray.get(i).getAsString().trim());

                    requirement.put(entry.getKey(), values);
                }
            }
        }

        parserConfig.setTableDescription(
            TableDescription.create(parseEngine(parser), parseColumns(parser))
        );

        return parserConfig;
    }

    public static JsonParserConfig getJsonParserConfig(JsonObject configObject) throws ConfigurationException {
        if (!configObject.has("jsonParser")) {
            return null;
        }
        JsonObject jsonParserObject = configObject.getAsJsonObject("jsonParser");
        return JsonParserConfig.fromJson(jsonParserObject, parseEngine(jsonParserObject));
    }


    private static EngineType parseEngine(JsonObject parser) {
        JsonObject tableEngine = parser.getAsJsonObject("tableEngine");

        if (tableEngine == null) {
            return TableDescription.DEFAULT_ENGINE;
        }

        String partitionBy = Optional.ofNullable(tableEngine.getAsJsonPrimitive("partitionBy"))
            .map(JsonElement::getAsString)
            .orElse(TableDescription.DEFAULT_ENGINE.getPartitionBy());

        List<String> orderBy = Optional.ofNullable(tableEngine.getAsJsonArray("orderBy"))
            .map(json ->
                StreamSupport.stream(json.spliterator(), false)
                    .map(JsonElement::getAsString)
                    .collect(Collectors.toList())
            )
            .orElse(TableDescription.DEFAULT_ENGINE.getOrderBy());

        String sampleBy = Optional.ofNullable(tableEngine.getAsJsonPrimitive("sampleBy"))
            .map(JsonElement::getAsString)
            .orElse(TableDescription.DEFAULT_ENGINE.getSampleBy());

        return new MergeTree(partitionBy, orderBy, sampleBy);
    }

    private static List<Column> parseColumns(JsonObject parser) {
        JsonObject columnsObject = parser.getAsJsonObject("columns");
        List<Column> columns = new ArrayList<>(columnsObject.entrySet().size());
        for (Map.Entry<String, JsonElement> entry : columnsObject.entrySet()) {
            String name = entry.getKey();
            String type = entry.getValue().getAsJsonObject().get("type").getAsString();

            JsonElement element = entry.getValue().getAsJsonObject().get("defaultExpr");
            Column column = new Column(
                name, ColumnTypeUtils.fromClickhouseDDL(type), element != null ? element.getAsString() : null);
            element = entry.getValue().getAsJsonObject().get("default");
            if (element != null) {
                column.setDefaultValue(element.getAsString());
            }
            columns.add(column);
        }
        return columns;
    }

    private static Map<String, String> readParams(JsonObject configObject) {
        if (!configObject.has("params")) {
            return Collections.emptyMap();
        }

        Map<String, String> params = new HashMap<>();
        JsonObject paramsObject = configObject.getAsJsonObject("params");
        for (Map.Entry<String, JsonElement> paramEntry : paramsObject.entrySet()) {
            params.put(paramEntry.getKey(), paramEntry.getValue().getAsString());
        }
        return Collections.unmodifiableMap(params);
    }

    public String getDefaultClickHouseDatabase() {
        return defaultClickHouseDatabase;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (zookeeperQuorum != null) {
            curatorFramework = CuratorFrameworkFactory.builder()
                .connectString(zookeeperQuorum)
                .retryPolicy(new RetryNTimes(Integer.MAX_VALUE, zookeeperTimeoutMillis))
                .connectionTimeoutMs(zookeeperTimeoutMillis)
                .sessionTimeoutMs(zookeeperTimeoutMillis)
                .namespace(logshatterZookeeperPrefix)
                .build();

            log.info("Connecting to zookeeper quorum {}, prefix {}", zookeeperQuorum, logshatterZookeeperPrefix);

            curatorFramework.start();
        }

        this.configs = Collections.unmodifiableList(readConfiguration());
        Multimap<LogSource, LogShatterConfig> newSourceToConfigs = ArrayListMultimap.create();
        for (LogShatterConfig config : configs) {
            for (LogSource source : config.getSources()) {
                newSourceToConfigs.put(source, config);
            }
        }
        this.sourceToConfigs = Multimaps.unmodifiableMultimap(newSourceToConfigs);
        checkDDL();
        initDDLWorker();
        checkTablesWithLock();
    }

    public void checkDDL() {
        MultiMap<String, LogShatterConfig> tableToConfig = new MultiMap<>();

        for (LogShatterConfig logShatterConfig : configs) {
            tableToConfig.append(logShatterConfig.getTableName(), logShatterConfig);
        }

        for (String table : tableToConfig.keySet()) {
            checkTableName(table);

            List<LogShatterConfig> tableConfigs = tableToConfig.get(table);
            LogShatterConfig firstConfig = tableConfigs.get(0);
            TableDescription firstDescription = firstConfig.createParser().getTableDescription();

            for (int i = 1; i < tableConfigs.size(); i++) {
                LogShatterConfig otherConfig = tableConfigs.get(i);
                TableDescription otherDescription = otherConfig.createParser().getTableDescription();

                if (!firstDescription.equals(otherDescription)) {
                    throw new IllegalStateException(
                        "Wrong configuration! One table with different descriptions.\n" +
                            "First config: " + firstConfig.getConfigFileName() +
                            ", parser: " + firstConfig.getParserClassName() +
                            ", description: \"" + firstDescription + "\".\n" +
                            "Second config: " + otherConfig.getConfigFileName() +
                            ", parser: " + otherConfig.getParserClassName() +
                            ", description: \"" + otherDescription + "\"."
                    );
                }
            }
        }
    }

    public static void checkTableName(String tableName) {
        if (!TABLE_NAME_REXEXP.matcher(tableName).matches()) {
            throw new IllegalStateException(
                "Wrong table name: " + tableName + ". Identifiers must match the regex " + TABLE_NAME_REXEXP
            );
        }
    }

    public void checkTablesWithLock() throws Exception {
        ddlWorker.run();
        ddlWorker.awaitStatus();
    }

    public List<LogShatterConfig> getConfigs() {
        return configs;
    }

    @VisibleForTesting
    public void setConfigs(List<LogShatterConfig> configs) {
        this.configs = configs;
    }

    public Multimap<LogSource, LogShatterConfig> getSourceToConfigs() {
        return sourceToConfigs;
    }

    public Multimap<LogSource, LogShatterConfig> getConfigsForSchema(String schema) {
        Multimap<LogSource, LogShatterConfig> schemaConfigs = ArrayListMultimap.create();
        for (LogSource source : sourceToConfigs.keySet()) {
            if (source.getSchema().equals(schema)) {
                schemaConfigs.putAll(source, sourceToConfigs.get(source));
            }
        }
        return schemaConfigs;
    }

    // public for tests
    public List<LogShatterConfig> readConfiguration() {
        File[] files = getConfigFiles();
        if (files == null || files.length == 0) {
            throw new RuntimeException("No configs found in dir:" + configDir.getPath());
        }
        log.info("Found " + files.length + " in dir: " + configDir.getPath());
        List<LogShatterConfig> newConfigs = new ArrayList<>();
        for (File file : files) {
            try {
                newConfigs.add(readFile(file));
            } catch (Exception e) {
                throw new RuntimeException("Failed to read config file: " + file.getPath(), e);
            }
        }
        return newConfigs;
    }

    private File[] getConfigFiles() {
        return configDir.listFiles((dir, name) -> name.toLowerCase().endsWith(".json"));
    }

    // public for tests
    public LogShatterConfig readFile(File file) throws IOException, ConfigValidationException, ConfigurationException {
        String data = FileUtils.readFileToString(file);
        validateJsonSchema(data);

        JsonObject configObject = JSON_PARSER.parse(data).getAsJsonObject();
        Path logDir = null;
        if (configObject.has("logDir")) {
            logDir = Paths.get(configObject.getAsJsonPrimitive("logDir").getAsString());
        }
        List<LogSource> sources = defaultSources;
        if (configObject.has("sources")) {
            sources = new ArrayList<>();
            JsonArray identsArray = configObject.getAsJsonArray("sources");
            for (JsonElement identElement : identsArray) {
                sources.add(LogSource.create(identElement.getAsString()));
            }
        }

        ParserConfig parserConfig = getParserConfig(configObject);
        String parserClassName = getParserClass(configObject);
        JsonParserConfig jsonParserConfig = getJsonParserConfig(configObject);
        LogParserProvider logParserProvider = new LogParserProvider(parserClassName, parserConfig, jsonParserConfig);


        String configTableName = configObject.getAsJsonPrimitive("clickhouseTable").getAsString().trim();
        String databaseName;
        String tableName;
        if (configTableName.contains(".")) {
            String[] splits = configTableName.split("\\.", 2);
            databaseName = splits[0];
            tableName = splits[1];
        } else {
            databaseName = defaultClickHouseDatabase;
            tableName = configTableName;
        }

        TableDescription tableDescription = logParserProvider.createParser().getTableDescription();

        ClickHouseTableDefinition dataTableDefinition;
        ClickHouseTableDefinition distributedTableDefinition;

        if (clickhouseDdlService.isDistributed()) {
            String dataTableName = tableName + LOCAL_REPLICATED_TABLE_POSTFIX;
            dataTableDefinition = new ClickHouseTableDefinitionImpl(
                databaseName,
                dataTableName,
                tableDescription.getColumns(),
                tableDescription.getEngine().replicated(databaseName, dataTableName, clickHouseZookeeperTablePrefix)
            );
            distributedTableDefinition = new ClickHouseTableDefinitionImpl(
                databaseName,
                tableName,
                tableDescription.getColumns(),
                new DistributedEngine(
                    clickhouseDdlService.getCluster(), databaseName, dataTableName, "rand()"
                )
            );
        } else {
            dataTableDefinition = new ClickHouseTableDefinitionImpl(
                databaseName,
                tableName,
                tableDescription.getColumns(),
                tableDescription.getEngine()
            );
            distributedTableDefinition = null;
        }

        final int dataRotationDays = (configObject.has("dataRotationDays"))
            ? configObject.getAsJsonPrimitive("dataRotationDays").getAsInt()
            : defaultDataRotationDays;

        return LogShatterConfig.newBuilder()
            .setLogDir(logDir)
            .setSources(sources)
            .setDistributedClickHouseTable(distributedTableDefinition)
            .setDataClickHouseTable(dataTableDefinition)
            .setParserProvider(logParserProvider)
            .setConfigFileName(file.getName())
            .setLogHosts(configObject.getAsJsonPrimitive("logHosts").getAsString().trim())
            .setLogPath(configObject.getAsJsonPrimitive("logPath").getAsString().trim())
            .setParams(readParams(configObject))
            .setDataRotationDays(dataRotationDays)
            .build();
    }


    private String getParserClass(JsonObject configObject) {
        if (!configObject.has("parserClass")) {
            return null;
        }
        return configObject.getAsJsonPrimitive("parserClass").getAsString().trim();
    }

    public void validateJsonSchema(String data) throws IOException, ConfigValidationException {
        JsonNode dataNode = NODE_READER.readTree(new StringReader(data));

        ProcessingReport report = VALIDATOR.validateUnchecked(VALIDATION_SCHEMA, dataNode);
        if (!report.isSuccess()) {
            throw new ConfigValidationException(JacksonUtils.prettyPrint(((AsJson) report).asJson()));
        }
    }

    private void initDDLWorker() throws Exception {
        MonitoringUnit ddlUpdateMonitoring = new MonitoringUnit("DDL update");
        monitoring.getHostCritical().addUnit(ddlUpdateMonitoring);
        ddlWorker = new UpdateDDLWorker(curatorFramework, updateDDLService, configs, ddlUpdateMonitoring, httpPort);
    }

    public UpdateDDLWorker getDDLWorker() {
        return ddlWorker;
    }

    @Required
    public void setDefaultClickHouseDatabase(String defaultClickHouseDatabase) {
        this.defaultClickHouseDatabase = defaultClickHouseDatabase;
    }

    @Required
    public void setClickhouseDdlService(ClickHouseDdlServiceOld clickhouseDdlService) {
        this.clickhouseDdlService = clickhouseDdlService;
    }

    @Required
    public void setConfigDir(String configDir) {
        this.configDir = new File(configDir);
    }

    @Required
    public void setDefaultSource(String defaultSource) {
        List<String> sources = Splitter.on(',').trimResults().omitEmptyStrings().splitToList(defaultSource);
        this.defaultSources = new ArrayList<>(sources.size());
        for (String source : sources) {
            try {
                defaultSources.add(LogSource.create(source));
            } catch (ConfigValidationException e) {
                throw new IllegalArgumentException(e);
            }
        }
    }

    @Required
    public void setClickHouseZookeeperTablePrefix(String clickHouseZookeeperTablePrefix) {
        this.clickHouseZookeeperTablePrefix = clickHouseZookeeperTablePrefix;
    }

    @Required
    public void setLogshatterZookeeperPrefix(String logshatterZookeeperPrefix) {
        this.logshatterZookeeperPrefix = logshatterZookeeperPrefix;
    }

    public void setZookeeperQuorum(String zookeeperQuorum) {
        this.zookeeperQuorum = zookeeperQuorum;
    }

    public void setUpdateDDLService(UpdateDDLService updateDDLService) {
        this.updateDDLService = updateDDLService;
    }

    public void setMonitoring(LogShatterMonitoring monitoring) {
        this.monitoring = monitoring;
    }

    public void setDefaultDataRotationDays(int defaultDataRotationDays) {
        this.defaultDataRotationDays = defaultDataRotationDays;
    }

    public void setHttpPort(int httpPort) {
        this.httpPort = httpPort;
    }

    public UserAgentDetector getUserAgentDetector() {
        return userAgentDetector;
    }

    public void setUserAgentDetector(UserAgentDetector userAgentDetector) {
        this.userAgentDetector = userAgentDetector;
    }
}
