package ru.yandex.market.logshatter.config;

import com.google.common.collect.ImmutableSet;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinition;
import ru.yandex.market.logshatter.LogShatterUtil;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.LogParserProvider;

import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 19/02/15
 */
public class LogShatterConfig {
    private static final Set<String> ANY_WILD_CARDS = ImmutableSet.of("*", "**");
    private final String configFileName;
    private final List<LogSource> sources;
    private final String logHosts;
    private final String logPath;
    private final PathMatcher fileMatcher;
    private final PathMatcher hostMatcher;
    private final LogParserProvider parserProvider;
    private final ClickHouseTableDefinition distributedTable;
    private final ClickHouseTableDefinition dataTable;
    private final Map<String, String> params;
    private final String shortConfigName;
    private final int dataRotationDays;

    private LogShatterConfig(Builder builder) {
        configFileName = builder.configFileName;
        sources = builder.sources;
        parserProvider = builder.parserProvider;
        distributedTable = builder.distributedClickHouseTable;
        dataTable = builder.dataClickHouseTable;
        params = builder.params;

        logHosts = builder.logHosts;
        logPath = builder.logPath;

        fileMatcher = LogShatterUtil.createGlobMatcher(logPath);
        hostMatcher = LogShatterUtil.createGlobMatcher(logHosts);

        Path configPath = Paths.get(configFileName);
        shortConfigName = configPath.getName(configPath.getNameCount() - 1).toString().split("\\.")[0];

        dataRotationDays = builder.dataRotationDays;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getLogHosts() {
        return logHosts;
    }

    public boolean isLogHostsWild() {
        return ANY_WILD_CARDS.contains(logHosts);
    }

    public String getLogPath() {
        return logPath;
    }

    public boolean isLogPathWild() {
        return ANY_WILD_CARDS.contains(logPath);
    }

    public String getParserClassName() {
        return parserProvider.getName();
    }

    public String getShortParserName() {
        return parserProvider.getShortName();
    }

    public String getConfigFileName() {
        return configFileName;
    }

    public PathMatcher getFileMatcher() {
        return fileMatcher;
    }

    public ClickHouseTableDefinition getDistributedTable() {
        return distributedTable;
    }

    public ClickHouseTableDefinition getDataTable() {
        return dataTable;
    }

    public String getInsertTableName() {
        return dataTable.getFullTableName();
    }

    public String getTableName() {
        return distributedTable != null ? distributedTable.getFullTableName() : dataTable.getFullTableName();
    }

    public Map<String, String> getParams() {
        return params;
    }

    public String getShortConfigName() {
        return shortConfigName;
    }

    public List<LogSource> getSources() {
        return sources;
    }

    public String getSchema() {
        return sources.get(0).getSchema();
    }

    public PathMatcher getHostMatcher() {
        return hostMatcher;
    }

    public int getDataRotationDays() {
        return dataRotationDays;
    }

    public LogParserProvider getParserProvider() {
        return parserProvider;
    }

    public LogParser createParser() {
        return parserProvider.createParser();
    }

    @Override
    public String toString() {
        return "ConfigContext{" +
            "parserClass=" + getParserClassName() +
            ", clickHouseTable='" + getTableName() + '\'' +
            '}';
    }

    public static final class Builder {
        private String configFileName;
        private Path logDir;
        private LogParserProvider parserProvider;
        private ClickHouseTableDefinition distributedClickHouseTable;
        private ClickHouseTableDefinition dataClickHouseTable;
        private Map<String, String> params;
        private List<LogSource> sources;
        private String logHosts;
        private String logPath;
        private int dataRotationDays;

        private Builder() {
        }

        public Builder setConfigFileName(String configFileName) {
            this.configFileName = configFileName;
            return this;
        }

        public Builder setLogDir(Path logDir) {
            this.logDir = logDir;
            return this;
        }

        public Builder setParserProvider(LogParserProvider parserProvider) {
            this.parserProvider = parserProvider;
            return this;
        }

        public Builder setDistributedClickHouseTable(ClickHouseTableDefinition distributedClickHouseTable) {
            this.distributedClickHouseTable = distributedClickHouseTable;
            return this;
        }

        public Builder setDataClickHouseTable(ClickHouseTableDefinition dataClickHouseTable) {
            this.dataClickHouseTable = dataClickHouseTable;
            return this;
        }

        public Builder setParams(Map<String, String> params) {
            this.params = params;
            return this;
        }

        public Builder setSources(List<LogSource> sources) {
            this.sources = sources;
            return this;
        }

        public Builder setLogHosts(String logHosts) {
            this.logHosts = logHosts;
            return this;
        }

        public Builder setLogPath(String logPath) {
            this.logPath = logPath;
            return this;
        }

        public Builder setDataRotationDays(int dataRotationDays) {
            this.dataRotationDays = dataRotationDays;
            return this;
        }

        public LogShatterConfig build() {
            return new LogShatterConfig(this);
        }
    }
}
