package ru.yandex.market.logshatter.config;

import org.apache.commons.lang3.StringUtils;
import ru.yandex.common.util.collections.MultiMap;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.clickhouse.ddl.DdlQuery;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 27.05.2015
 */
public class LogshatterConfigLog {
    private static final String ERRORS = "-----------\nErrors occurred while updating tables!\n-----------\n";
    private static final String HEADER = "-----------\n-----------\nManual DDL required!\n-----------\n";
    private static final String FOR_EXECUTOR = "-----------\nFor executor:\n-----------\n";
    private static final String FOOTER =
        "-----------\nDo not forget to restart logshatter!\n-----------\n-----------\n\n\n\n";

    private static final String EXEC_FORMAT = "exec %s clickhouse-client -q \"%s\"";

    private LogshatterConfigLog() {
    }

    public static String format(List<DDL> tableDDLs) {
        MultiMap<String, String> manualUpdates = new MultiMap<>();
        MultiMap<String, String> errors = new MultiMap<>();
        for (DDL ddl : tableDDLs) {
            List<String> manualQueries = ddl.getManualUpdates()
                .stream()
                .map(DdlQuery::getQueryString)
                .collect(Collectors.toList());

            manualUpdates.put(ddl.getHost(), manualQueries);
            errors.put(ddl.getHost(), ddl.getErrors());
        }
        return format(manualUpdates, errors);
    }

    private static String format(MultiMap<String, String> manualUpdates, MultiMap<String, String> errors) {
        return "\n" + new Date() + "\n" + processErrors(errors) + processManualUpdates(manualUpdates);
    }

    private static String processErrors(MultiMap<String, String> errors) {
        if (errors.isEmpty()) {
            return "";
        }
        return ERRORS + getDescriptionPart(errors) + "\n\n\n";
    }

    private static String processManualUpdates(MultiMap<String, String> manualUpdates) {
        if (manualUpdates.isEmpty()) {
            return "";
        }
        return HEADER + getDescriptionPart(manualUpdates) + FOR_EXECUTOR + getExecutionPart(manualUpdates) + FOOTER;
    }

    private static String getDescriptionPart(MultiMap<String, String> manualUpdates) {
        StringBuilder sb = new StringBuilder();

        for (Map.Entry<String, List<String>> entry : manualUpdates.entrySet()) {
            List<String> queries = entry.getValue();
            if (queries == null || queries.isEmpty()) {
                continue;
            }

            sb.append(entry.getKey()).append(":\n").append(StringUtils.join(queries, ";\n")).append("\n");
        }
        return sb.toString();
    }

    private static String getExecutionPart(MultiMap<String, String> manualUpdates) {
        StringBuilder sb = new StringBuilder();

        for (Map.Entry<String, List<String>> entry : manualUpdates.entrySet()) {
            List<String> queries = entry.getValue();
            if (queries == null || queries.isEmpty()) {
                continue;
            }

            String host = entry.getKey();
            for (String query : queries) {
                sb.append(String.format(EXEC_FORMAT, host, query)).append("\n");
            }
        }
        return sb.toString();
    }
}
