package ru.yandex.market.logshatter.config.ddl;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickhouse.ClickHouseSource;
import ru.yandex.market.clickhouse.ddl.ClickHouseDdlServiceOld;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.clickhouse.ddl.DdlQuery;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.config.LogshatterConfigLog;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateHostDDLResult;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateHostDDLTask;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateHostDDLTaskFactory;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateShardDDLTask;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateShardDDLTaskImpl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.function.Consumer;
import java.util.stream.Collectors;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 02.11.16
 */
public class UpdateDDLService {
    private static final Logger log = LogManager.getLogger();
    private static final Logger configLog = LogManager.getLogger("config");

    private final UpdateHostDDLTaskFactory updateHostDDLTaskFactory;
    private final ClickHouseSource clickHouseSource;
    private final ClickHouseDdlServiceOld clickhouseDdlService;

    private final List<Consumer<UpdateDDLTaskExecutorResult>> subscribers = new ArrayList<>();
    private final ExecutorService ddlExecutorService;

    private volatile List<DDL> manualDDLs = Collections.emptyList();


    public UpdateDDLService(ClickHouseSource clickHouseSource, UpdateHostDDLTaskFactory updateHostDDLTaskFactory,
                            ClickHouseDdlServiceOld clickhouseDdlService, int threadCount) {
        this.clickHouseSource = clickHouseSource;
        this.updateHostDDLTaskFactory = updateHostDDLTaskFactory;
        this.clickhouseDdlService = clickhouseDdlService;

        ddlExecutorService = Executors.newFixedThreadPool(
            threadCount,
            new ThreadFactoryBuilder().setNameFormat("ddl-%d").build()
        );
    }

    public List<DDL> updateDDL(List<LogShatterConfig> configs) {
        log.info("Checking tables in clickhouse");

        Multimap<Integer, String> shard2Hosts = clickHouseSource.getShard2Hosts();
        if (shard2Hosts == null) {
            shard2Hosts = getOneShardWithOneHost(clickHouseSource.getHost());
        }

        UpdateDDLTaskExecutor taskExecutor = new UpdateDDLTaskExecutor();
        taskExecutor.onResult(this::logManualDDL);

        for (Consumer<UpdateDDLTaskExecutorResult> subscriber : subscribers) {
            taskExecutor.onResult(subscriber);
        }

        ListMultimap<String, LogShatterConfig> tableToConfigs = Multimaps.index(
            configs, LogShatterConfig::getTableName
        );

        for (String table : tableToConfigs.keySet()) {
            for (Integer shardNumber : shard2Hosts.keySet()) {
                UpdateShardDDLTask task = new UpdateShardDDLTaskImpl(
                    updateHostDDLTaskFactory,
                    shard2Hosts.get(shardNumber),
                    tableToConfigs.get(table)
                );
                taskExecutor.commitTask(task);
            }
        }
        return taskExecutor.allTasksCommitted();
    }


    public UpdateDDLOnHostResult applyDdlOnHost(List<LogShatterConfig> configs, String host, boolean applyManual) {
        ListMultimap<String, LogShatterConfig> tableToConfigs = Multimaps.index(
            configs, LogShatterConfig::getTableName
        );
        List<UpdateHostDDLTask> tasks = new ArrayList<>();
        for (String table : tableToConfigs.keySet()) {
            LogShatterConfig config = tableToConfigs.get(table).get(0);
            if (config.getDistributedTable() != null) {
                tasks.add(updateHostDDLTaskFactory.create(host, config.getDistributedTable()));
            }
            tasks.add(updateHostDDLTaskFactory.create(host, config.getDataTable()));
        }


        List<Future<UpdateHostDDLResult>> futureResults = tasks.stream()
            .map(task -> ddlExecutorService.submit(task::checkTableDDL))
            .collect(Collectors.toList());

        List<UpdateHostDDLResult> results = futureResults.stream()
            .map(updateHostDDLResultFuture -> {
                try {
                    return updateHostDDLResultFuture.get();
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            })
            .collect(Collectors.toList());

        List<DDL> manualDdls = new ArrayList<>();
        List<UpdateDDLException> updateDDLExceptions = new ArrayList<>();
        for (UpdateHostDDLResult result : results) {
            if (result instanceof UpdateHostDDLResult.ManualDDLRequired) {
                manualDdls.add(((UpdateHostDDLResult.ManualDDLRequired) result).getManualDDL());
            }
            if (result instanceof UpdateHostDDLResult.Error) {
                updateDDLExceptions.add(((UpdateHostDDLResult.Error) result).getException());
            }
        }
        ManualDDLExecutionResult manualDDLExecutionResult = null;
        if (!manualDdls.isEmpty() && applyManual) {
            manualDDLExecutionResult = executeManualDDLs(manualDdls);
        }
        return new UpdateDDLOnHostResult(manualDdls, manualDDLExecutionResult, updateDDLExceptions);
    }

    public void watchResult(Consumer<UpdateDDLTaskExecutorResult> callback) {
        subscribers.add(callback);
    }

    ManualDDLExecutionResult executeManualDDLs(List<DDL> ddlList) {
        List<DDL> failedDDLs = new ArrayList<>();
        List<DDL> succeededDDLs = new ArrayList<>();
        for (DDL ddl : ddlList) {
            try {
                clickhouseDdlService.executeManualDDL(ddl);
                succeededDDLs.add(ddl);
            } catch (Exception e) {
                String failingQueries = ddl.getManualUpdates().stream()
                    .map(DdlQuery::getQueryString)
                    .collect(Collectors.joining(", "));
                log.error("Exception while executing manual DDL updates: " + failingQueries, e);
                failedDDLs.add(ddl);
            }
        }

        // TODO: здесь возможна неконсистентность, если параллельно был запущен процесс апдейта DDL
        // Он может перезаписать результат вызова этого метода и пользователь при следующем вызове ручки getManualDDLs
        // снова получит те запросы, которые были только что выполнены
        // TODO: синхронизировать эти процессы
        this.manualDDLs = failedDDLs;
        return new ManualDDLExecutionResult(succeededDDLs, failedDDLs);
    }

    public List<DDL> getManualDDLs() {
        return manualDDLs;
    }

    private void logManualDDL(UpdateDDLTaskExecutorResult result) {
        if (!(result instanceof UpdateDDLTaskExecutorResult.ManualDDLRequired)) {
            manualDDLs = Collections.emptyList();
            return;
        }

        manualDDLs = ((UpdateDDLTaskExecutorResult.ManualDDLRequired) result).getManualDDLs();

        List<DDL> ddlList = manualDDLs;
        if (!ddlList.isEmpty()) {
            configLog.info(LogshatterConfigLog.format(ddlList));
        }
    }

    private static Multimap<Integer, String> getOneShardWithOneHost(String host) {
        Multimap<Integer, String> oneShardToOneHost = HashMultimap.create();
        oneShardToOneHost.put(0, host);
        return oneShardToOneHost;
    }
}
