package ru.yandex.market.logshatter.config.ddl;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateShardDDLResult;
import ru.yandex.market.logshatter.config.ddl.shard.UpdateShardDDLTask;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 31.10.16
 */
class UpdateDDLTaskExecutor {
    private static final Logger log = LogManager.getLogger();

    private int timeBetweenRetriesMillis = (int) TimeUnit.MINUTES.toMillis(1);

    private final List<Consumer<UpdateDDLTaskExecutorResult>> subscribers = new ArrayList<>();
    private List<UpdateShardDDLTask> queue;

    private boolean allTasksCommitted = false;

    UpdateDDLTaskExecutor() {
        this.queue = new ArrayList<>();
    }

    void commitTask(UpdateShardDDLTask task) {
        if (allTasksCommitted) {
            throw new IllegalStateException("Unable to commit task after allTasksCommitted() call");
        }

        queue.add(task);
    }

    List<DDL> allTasksCommitted() {
        allTasksCommitted = true;
        return this.processQueue();
    }

    void onResult(Consumer<UpdateDDLTaskExecutorResult> callback) {
        subscribers.add(callback);
    }

    void setTimeBetweenRetriesMillis(int timeBetweenRetriesMillis) {
        this.timeBetweenRetriesMillis = timeBetweenRetriesMillis;
    }

    private void notifySubscribers(UpdateDDLTaskExecutorResult result) {
        for (Consumer<UpdateDDLTaskExecutorResult> subscriber : subscribers) {
            subscriber.accept(result);
        }
    }

    private List<DDL> processQueue() {
        List<DDL> ddls = new ArrayList<>();

        if (queue.isEmpty()) {
            notifySubscribers(new UpdateDDLTaskExecutorResult.Success());
            return ddls;
        }

        while (!queue.isEmpty()) {
            ShardResultsView resultsView = new ShardResultsView(
                queue
                    .parallelStream()
                    .map(UpdateShardDDLTask::run)
                    .collect(Collectors.toList())
            );

            resultsView.getSuccessfulResults().stream()
                .map(UpdateShardDDLResult.Success::getDdls)
                .forEach(ddls::addAll);

            logConnectionExceptions(resultsView);

            List<UpdateShardDDLResult.ManualDDLRequired> manualDDLResults = resultsView.getManualDDLResults();

            if (!manualDDLResults.isEmpty()) {
                notifySubscribers(
                    new UpdateDDLTaskExecutorResult.ManualDDLRequired(
                        manualDDLResults
                            .stream()
                            .flatMap(r -> r.getManualDDLs().stream())
                            .collect(Collectors.toList())
                    )
                );
            } else if (areAllShardsInitialized(resultsView)) {
                if (!resultsView.getPartialSuccessResults().isEmpty()) {
                    notifySubscribers(new UpdateDDLTaskExecutorResult.PartialSuccess());
                } else {
                    notifySubscribers(new UpdateDDLTaskExecutorResult.Success());
                }
            }

            queue = resultsView.getUnsuccessfulResults()
                .stream()
                .map(UpdateShardDDLResult::getTask)
                .collect(Collectors.toList());

            if (!queue.isEmpty()) {
                try {
                    log.info(
                        "There are {} unsuccessful tasks, sleeping for {}ms before retry",
                        queue.size(),
                        timeBetweenRetriesMillis
                    );

                    Thread.sleep(timeBetweenRetriesMillis);
                } catch (InterruptedException e) {
                    return ddls;
                }
            }
        }

        return ddls;
    }

    private boolean areAllShardsInitialized(ShardResultsView shardResults) {
        return shardResults.getFailedResults().isEmpty();
    }

    private void logConnectionExceptions(ShardResultsView shardResults) {
        List<UpdateDDLException> exceptions = Stream.concat(
            shardResults.getPartialSuccessResults().stream().flatMap(r -> r.getExceptions().stream()),
            shardResults.getFailedResults().stream().flatMap(r -> r.getExceptions().stream())
        ).collect(Collectors.toList());

        Set<String> loggedHosts = new HashSet<>();

        for (UpdateDDLException exception : exceptions) {
            String host = exception.getHost();
            if (loggedHosts.contains(host)) {
                continue;
            }

            log.error("Excepton while connecting to " + host, exception.getCause());
            loggedHosts.add(host);
        }
    }
}
