package ru.yandex.market.logshatter.config.ddl;

import com.google.common.util.concurrent.SettableFuture;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.PathChildrenCache;
import org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent;
import org.apache.curator.framework.recipes.leader.LeaderSelector;
import org.apache.curator.framework.recipes.leader.LeaderSelectorListener;
import org.apache.curator.framework.state.ConnectionState;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.monitoring.MonitoringUnit;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import static org.apache.curator.framework.recipes.cache.PathChildrenCacheEvent.Type.*;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 02.11.16
 * Общается с зукипером и понимает, когда можно дать отмашку о том, что можно начинать процессить логи
 */
public class UpdateDDLWorker {
    private static final String LEADER_PATH = "/ddlMaster";
    private static final String STATUS_NODE_PATH = LEADER_PATH + "/status";

    private static final Logger log = LogManager.getLogger();

    private final LeaderSelector leaderSelector;
    private final CuratorFramework curatorFramework;
    private final SettableFuture<UpdateDDLStatus> future = SettableFuture.create();
    private final PathChildrenCache cache;
    private final MonitoringUnit monitoringUnit;
    private final Thread whoIsMasterThread;
    private final CountDownLatch leaderCanFinishLatch = new CountDownLatch(1);
    private final UpdateDDLService ddlService;
    private final int httpPort;

    private volatile boolean isFinished = false;
    private volatile int timeToWaitFollowersToGetStatusMillis = (int) TimeUnit.MINUTES.toMillis(1);

    public UpdateDDLWorker(CuratorFramework curatorFramework, UpdateDDLService ddlService,
                           List<LogShatterConfig> configs, MonitoringUnit monitoringUnit,
                           int httpPort) throws Exception {
        this.ddlService = ddlService;
        this.curatorFramework = curatorFramework;
        this.cache = new PathChildrenCache(curatorFramework, LEADER_PATH, false);
        this.monitoringUnit = monitoringUnit;
        this.httpPort = httpPort;

        cache.getListenable().addListener(this::onChildEvent);
        cache.start();

        this.leaderSelector = new LeaderSelector(curatorFramework, LEADER_PATH, new LeaderSelectorListener() {
            @Override
            public void takeLeadership(CuratorFramework client) throws Exception {
                try {
                    log.info("Got DDL leadership, Starting DDL update...");
                    updateDDL(ddlService, curatorFramework, configs);

                    // ждём обратной связи о том, что статус выставлен, и можно завершать работу
                    leaderCanFinishLatch.await();
                    log.info("Leader finishing...");
                } catch (Exception e) {
                    monitoringUnit.critical("DDL update finished unsuccessfully", e);
                    log.error("DDL Service has thrown exception {}", e);
                    future.setException(e);
                    throw e;
                }
            }

            @Override
            public void stateChanged(CuratorFramework client, ConnectionState newState) {
                log.info("Curator connection state changed to {}", newState);
            }
        });

        this.whoIsMasterThread = new Thread(() -> {
            while (!Thread.interrupted()) {
                logCurrentMaster();
                try {
                    TimeUnit.MINUTES.sleep(2);
                } catch (InterruptedException ignored) {
                }
            }
        });

        whoIsMasterThread.setDaemon(true);

        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            try {
                log.info("Terminating...");
                // Если логшаттер попросили завершиться, нужно удалить эфемерные ноды
                curatorFramework.close();
            } catch (Exception e) {
                log.error("Shutdown hook exception", e);
            }
        }));
    }

    private void updateDDL(UpdateDDLService ddlService,
                           CuratorFramework curatorFramework,
                           List<LogShatterConfig> configs) {
        ddlService.watchResult(r -> {
            UpdateDDLStatus status = r.asStatus();
            reportToMonitoring(status);
            log.info("DDL Service returned status {}", status);
            byte[] statusBytes = status.toString().getBytes(StandardCharsets.UTF_8);
            try {
                Stat existingStatusNodeStat = curatorFramework.checkExists()
                    .creatingParentContainersIfNeeded()
                    .forPath(STATUS_NODE_PATH);
                if (existingStatusNodeStat != null) {
                    log.info("Existing node stat: " + existingStatusNodeStat);
                    curatorFramework.setData().forPath(STATUS_NODE_PATH, statusBytes);
                } else {
                    curatorFramework.create().creatingParentContainersIfNeeded()
                        .withMode(CreateMode.EPHEMERAL)
                        .forPath(STATUS_NODE_PATH, statusBytes);
                }
            } catch (Exception e) {
                throw new RuntimeException(e.getMessage(), e);
            }
        });

        ddlService.updateDDL(configs);
    }

    public void run() throws Exception {
        leaderSelector.start();
        whoIsMasterThread.start();
    }

    public UpdateDDLStatus awaitStatus() throws Exception {
        return future.get();
    }

    private void onChildEvent(CuratorFramework client, PathChildrenCacheEvent event) throws Exception {
        PathChildrenCacheEvent.Type eventType = event.getType();
        if (!(eventType == CHILD_ADDED || eventType == CHILD_UPDATED || eventType == INITIALIZED)) {
            return;
        }

        ChildData childData = event.getData();
        if (!childData.getPath().equals(STATUS_NODE_PATH)) {
            return;
        }

        try {
            log.info("Status node changed ({}), processing status...", eventType);

            byte[] data = curatorFramework.getData().forPath(STATUS_NODE_PATH);

            if (data == null) {
                // Это может означать, что пока мы ходили за нодой, мастер умер и нода отвалилась
                // Ждём следующего мастера, который выставит статус
                return;
            }

            UpdateDDLStatus status = UpdateDDLStatus.valueOf(new String(data, "UTF-8"));
            log.info("Status is {}", status);
            future.set(status);

            if (status.equals(UpdateDDLStatus.SUCCESS)) {
                finish();
            }
        } catch (Exception e) {
            future.setException(e);
        }
    }

    private void reportToMonitoring(UpdateDDLStatus status) {
        switch (status) {
            case SUCCESS:
                monitoringUnit.ok();
                break;
            case PARTIAL_SUCCESS:
                monitoringUnit.warning("DDL has been applied, but some hosts were unavailable. " +
                    "Look for UpdateHostDDLTaskImpl in /var/log/logshatter/logshatter.log");
                break;
            case COLUMN_DROPS_REQUIRED:
            case MANUAL_DDL_REQUIRED:
                String host;
                try {
                    host = InetAddress.getLocalHost().getCanonicalHostName();
                } catch (UnknownHostException e) {
                    throw new RuntimeException(e);
                }
                String getManualDDLUrl = String.format("http://%s:%d/getManualDDL", host, httpPort);
                monitoringUnit.critical("Manual DDL required!" +
                    " Follow " + getManualDDLUrl + " and then follow confirmation URL that will be given in response." +
                    " See https://wiki.yandex-team.ru/market/development/health/logshatter/#avtomaticheskoeprimenenieddl");
                break;
        }
    }

    public boolean isFinished() {
        return isFinished;
    }

    public void setTimeToWaitFollowersToGetStatusMillis(int timeToWaitFollowersToGetStatusMillis) {
        this.timeToWaitFollowersToGetStatusMillis = timeToWaitFollowersToGetStatusMillis;
    }

    private void finish() throws Exception {
        if (leaderSelector.hasLeadership()) {
            // ждём, пока все получат оповещение об успехе
            Thread.sleep(timeToWaitFollowersToGetStatusMillis);
            // и разрешаем мастеру завершиться
            leaderCanFinishLatch.countDown();

            // ждём, пока лидер получит нотификацию и закончит работу
            while (leaderSelector.hasLeadership()) {
            }
        }

        whoIsMasterThread.interrupt();
        leaderSelector.close();
        cache.close();
        curatorFramework.close();
        isFinished = true;
        log.info("finished");
    }

    private void logCurrentMaster() {
        if (leaderSelector == null) {
            return;
        }
        try {
            log.info("Current master is: " + leaderSelector.getLeader().getId());
        } catch (Exception ignored) {
        }
    }

    public List<DDL> getManualDDLs() {
        ensureLeadership();
        return ddlService.getManualDDLs();
    }

    public ManualDDLExecutionResult executeManualDDLs(List<DDL> manualDDLs) {
        ensureLeadership();
        return ddlService.executeManualDDLs(manualDDLs);
    }

    private void ensureLeadership() {
        if (!leaderSelector.hasLeadership()) {
            String errorMessage;
            try {
                errorMessage = "Current host is not DDL master. Master is " + leaderSelector.getLeader().getId();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
            throw new IllegalStateException(errorMessage);
        }
    }
}
