package ru.yandex.market.logshatter.config.ddl.shard;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickhouse.ClickhouseColumnAlreadyExistsException;
import ru.yandex.market.clickhouse.ClickhouseReplicaAlterTimeoutException;
import ru.yandex.market.clickhouse.ClickhouseTemplate;
import ru.yandex.market.clickhouse.ddl.ClickHouseDdlServiceOld;
import ru.yandex.market.clickhouse.ddl.ClickHouseTableDefinition;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.clickhouse.ddl.DdlQuery;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLException;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 27.01.17
 */
class UpdateHostDDLTaskImpl implements UpdateHostDDLTask {
    private static final Logger log = LogManager.getLogger();

    private final String host;
    private final ClickHouseTableDefinition tableDefinition;
    private final ClickHouseDdlServiceOld clickhouseDdlService;
    private ClickhouseTemplate clickhouseTemplate;

    public UpdateHostDDLTaskImpl(String host, ClickHouseTableDefinition tableDefinition, ClickHouseDdlServiceOld clickhouseDdlService,
                                 ClickhouseTemplate clickhouseTemplate) {
        this.host = host;
        this.tableDefinition = tableDefinition;
        this.clickhouseDdlService = clickhouseDdlService;
        this.clickhouseTemplate = clickhouseTemplate;
    }

    @Override
    public UpdateHostDDLResult checkTableDDL() {
        try {
            createDatabase();
            DDL ddl = clickhouseDdlService.getDDL(tableDefinition, host);

            if (ddl.requireAttention()) {
                return new UpdateHostDDLResult.ManualDDLRequired(ddl);
            }

            for (DdlQuery query : ddl.getUpdates()) {
                try {
                    clickhouseTemplate.update(query.getQueryString(), ddl.getHost());
                    log.info("DDL query applied successfully, host: {}, query: {}", host, query.getQueryString());
                } catch (ClickhouseReplicaAlterTimeoutException exception) {
                    // не удалось обновить DDL на одной из реплик -- считаем за успех, Clickhouse сам докатит
                    log.warn(
                        String.format(
                            "Unable to update Clickhouse replica, query: %s, host: %s", query.getQueryString(), host
                        ),
                        exception
                    );
                } catch (ClickhouseColumnAlreadyExistsException exception) {
                    log.warn(
                        String.format(
                            "It seems Clickhouse updated replica column by itself, query: %s, host: %s",
                            query.getQueryString(), host
                        ),
                        exception
                    );
                }
            }

            return new UpdateHostDDLResult.Success(ddl);
        } catch (Exception e) {
            log.error("Exception while applying ddl on host " + host, e);
            return new UpdateHostDDLResult.Error(new UpdateDDLException(e, host));
        }
    }

    private void createDatabase() {
        clickhouseTemplate.update(
            "CREATE DATABASE IF NOT EXISTS " + tableDefinition.getDatabaseName(),
            host,
            true
        );
    }

    @Override
    public String toString() {
        return "UpdateHostTask{" +
            "host='" + host + '\'' +
            '}';
    }
}
