package ru.yandex.market.logshatter.config.ddl.shard;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.clickhouse.ddl.DDL;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.config.ddl.UpdateDDLException;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.stream.Collectors;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 31.10.16
 */
public class UpdateShardDDLTaskImpl implements UpdateShardDDLTask {
    private static final Logger log = LogManager.getLogger();

    private Queue<UpdateHostDDLTask> hostTasks;

    public UpdateShardDDLTaskImpl(UpdateHostDDLTaskFactory updateHostDDLTaskFactory,
                                  Collection<String> shardHosts,
                                  List<LogShatterConfig> tableConfigs) {
        this.hostTasks = new ArrayDeque<>();

        for (String host : shardHosts) {
            LogShatterConfig config = tableConfigs.get(0);
            if (config.getDistributedTable() != null) {
                hostTasks.add(updateHostDDLTaskFactory.create(host, config.getDistributedTable()));
            }
            hostTasks.add(updateHostDDLTaskFactory.create(host, config.getDataTable()));

        }
    }

    public UpdateShardDDLResult run() {
        List<UpdateHostDDLResult> hostResults = new ArrayList<>();
        Queue<UpdateHostDDLTask> unsuccessfulTasks = new ArrayDeque<>();

        for (UpdateHostDDLTask task : hostTasks) {
            UpdateHostDDLResult result = task.checkTableDDL();
            if (!(result instanceof UpdateHostDDLResult.Success)) {
                unsuccessfulTasks.add(task);
            }

            hostResults.add(result);
        }

        hostTasks = unsuccessfulTasks;
        return aggregateResult(hostResults);
    }

    private UpdateShardDDLResult aggregateResult(List<UpdateHostDDLResult> hostResults) {
        List<DDL> manualDDLs = getResultsOfType(UpdateHostDDLResult.ManualDDLRequired.class, hostResults)
            .stream()
            .map(UpdateHostDDLResult.ManualDDLRequired::getManualDDL)
            .collect(Collectors.toList());

        if (!manualDDLs.isEmpty()) {
            return new UpdateShardDDLResult.ManualDDLRequired(this, manualDDLs);
        }

        List<UpdateDDLException> exceptions =
            this.getResultsOfType(UpdateHostDDLResult.Error.class, hostResults)
                .stream()
                .map(UpdateHostDDLResult.Error::getException)
                .collect(Collectors.toList());

        List<UpdateHostDDLResult.Success> successes = getResultsOfType(UpdateHostDDLResult.Success.class, hostResults);
        if (!exceptions.isEmpty()) {
            if (successes.isEmpty()) {
                return new UpdateShardDDLResult.Failure(this, exceptions);
            }
            return new UpdateShardDDLResult.PartialSuccess(this, exceptions);
        }

        return new UpdateShardDDLResult.Success(
            this,
            successes.stream()
                .map(UpdateHostDDLResult.Success::getDdl)
                .collect(Collectors.toList())
        );
    }

    private <T> List<T> getResultsOfType(Class<T> clazz, List<UpdateHostDDLResult> allResults) {
        return allResults.stream().filter(clazz::isInstance).map(clazz::cast).collect(Collectors.toList());
    }

    @Override
    public String toString() {
        return "UpdateShardDDLTask{" +
            ", hostTasks=" + hostTasks.stream().map(Object::toString).collect(Collectors.joining(", ")) +
            '}';
    }
}
