package ru.yandex.market.logshatter.generic.json;

import com.google.common.base.Preconditions;
import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 30/10/2017
 */
public class JsonParser implements LogParser {

    private final JsonParserConfig config;
    private final Gson gson = new Gson();
    private final List<JsonParserConfig.Column> columns;
    private final TableDescription tableDescription;
    private final DateFormat dateFormat;

    public static final String HOST_FIELD = "context:host";
    public static final String CURRENT_DATE_FIELD = "context:now";

    public JsonParser(JsonParserConfig config) {
        this.config = config;
        this.columns = config.getColumns();
        this.tableDescription = createTableDescription(config);
        this.dateFormat = config.getDateFormat();
    }

    private TableDescription createTableDescription(JsonParserConfig config) {
        List<Column> tableColumns = config.getColumns().stream()
            .map(c -> new Column(c.getName(), c.getType(), c.getDefaultExpr()))
            .collect(Collectors.toList());
        return TableDescription.create(config.getEngineType(), tableColumns);
    }

    @Override
    public TableDescription getTableDescription() {
        return tableDescription;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject jsonObject = gson.fromJson(line, JsonObject.class);

        Object[] values = new Object[columns.size()];

        Date date = getDate(jsonObject);

        for (int i = 0; i < columns.size(); i++) {
            JsonParserConfig.Column column = columns.get(i);
            values[i] = getValue(context, column, jsonObject);
        }
        context.write(date, values);
    }

    private Date getDate(JsonObject jsonObject) throws Exception {
        if (config.getTimestampFiled() != null || config.getTimestampJsonPath() != null) {
            Date date = getTimestampField(jsonObject);
            if (date != null) {
                return date;
            }
        }
        if (config.getDateField() != null || config.getDateJsonPath() != null) {
            Date date = getDateField(jsonObject);
            if (date != null) {
                return date;
            }
        }
        throw new ParserException("Field or timestamp not found in json: " + jsonObject.toString());
    }

    private Date getDateField(JsonObject jsonObject) throws ParseException {
        JsonElement jsonElement = getElement(jsonObject, config.getDateField(), config.getDateJsonPath());
        if (jsonElement == null) {
            return null;
        }
        return dateFormat.parse(jsonElement.getAsString());
    }

    private Date getTimestampField(JsonObject jsonObject) {
        JsonElement timestampElement = getElement(jsonObject, config.getTimestampFiled(), config.getTimestampJsonPath());
        if (timestampElement == null) {
            return null;
        }
        long timestamp = timestampElement.getAsLong();
        if (timestamp > Integer.MAX_VALUE) {
            return new Date(timestamp);
        } else {
            return new Date(TimeUnit.SECONDS.toMillis(timestamp));
        }

    }

    private Object getValue(ParserContext context, JsonParserConfig.Column column, JsonObject jsonObject) {
        String field = column.getField();
        if (HOST_FIELD.equals(field)) {
            return context.getHost();
        }
        if (CURRENT_DATE_FIELD.equals(field)) {
            return new Date();
        }

        JsonElement element = getElement(jsonObject, field, column.getJsonPath());
        if (element == null || element.isJsonNull()) {
            Preconditions.checkArgument(
                column.getDefaultValue() != null,
                "Field '%s' is undefined and no default value provided for column '%s'. Json: %s",
                column.getField(), column.getName(), jsonObject
            );
            return column.getDefaultValue();
        }
        String stringValue = element.isJsonPrimitive()
            ? element.getAsString()
            : element.toString();
        if (stringValue.isEmpty() && !column.getType().equals(ColumnType.String)) {
            Preconditions.checkArgument(
                column.getDefaultValue() != null,
                "Field '%s' is empty string and no default value provided for column '%s'. Json: %s",
                column.getField(), column.getName(), jsonObject
            );
            return column.getDefaultValue();

        }

        //TODO optimize for JsonData types
        return column.getType().parseValue(stringValue, null);
    }

    private JsonElement getElement(JsonObject jsonObject, String field, List<String> path) {
        if (field != null) {
            return jsonObject.get(field);
        }
        JsonElement el = jsonObject;
        for (String pathFragment : path) {
            if (el == null) {
                return null;
            }
            if (el.isJsonObject()) {
                el = el.getAsJsonObject().get(pathFragment);
            } else {
                return null;
            }
        }
        return el;
    }
}
