package ru.yandex.market.logshatter.generic.json;

import com.google.common.base.Preconditions;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import ru.yandex.market.clickhouse.ddl.ColumnTypeBase;
import ru.yandex.market.clickhouse.ddl.ColumnTypeUtils;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;

import javax.naming.ConfigurationException;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 30/10/2017
 */
public class JsonParserConfig {
    private final String dateFormat;
    private final String timestampFiled;
    private final List<String> timestampJsonPath;
    private final String dateField;
    private final List<String> dateJsonPath;
    private final List<Column> columns;
    private final EngineType engineType;

    public JsonParserConfig(String dateFormat, String timestampField, String timestampJsonPath, String dateField, String dateJsonPath, List<Column> columns,
                            EngineType engineType) {
        this.engineType = engineType;
        Preconditions.checkArgument(
            timestampField != null || dateField != null || timestampJsonPath != null || dateJsonPath != null,
            "Any of timestampField, timestampJsonPath, dateField or dateJsonPath should be set"
        );
        Preconditions.checkArgument(!columns.isEmpty(), "Columns are empty");
        Preconditions.checkArgument(
            (dateField == null && dateJsonPath == null) || dateFormat != null,
            "dateField or dateJsonPath is set, but no dateFormat is not provided"
        );
        this.dateFormat = dateFormat;
        this.timestampFiled = timestampField;
        this.timestampJsonPath = splitOptionalString(timestampJsonPath);
        this.dateField = dateField;
        this.dateJsonPath = splitOptionalString(dateJsonPath);
        this.columns = Collections.unmodifiableList(columns);
    }

    public DateFormat getDateFormat() {
        return dateFormat == null ? null : new SimpleDateFormat(dateFormat); //New instance for synchronization
    }

    public String getTimestampFiled() {
        return timestampFiled;
    }

    public List<String> getTimestampJsonPath() {
        return timestampJsonPath;
    }

    public String getDateField() {
        return dateField;
    }

    public List<String> getDateJsonPath() {
        return dateJsonPath;
    }

    public List<Column> getColumns() {
        return columns;
    }

    public EngineType getEngineType() {
        return engineType;
    }

    public static class Column {
        private final String name;
        private final ColumnTypeBase type;
        private final String field;
        private final List<String> jsonPath;
        private final Object defaultValue;
        private final String defaultExpr;

        public Column(String name, ColumnTypeBase type, String field, String jsonPath, Object defaultValue, String defaultExpr) {
            this.name = name;
            this.type = type;
            this.field = field;
            this.jsonPath = splitOptionalString(jsonPath);
            this.defaultValue = defaultValue;
            this.defaultExpr = defaultExpr;
        }

        public String getName() {
            return name;
        }

        public ColumnTypeBase getType() {
            return type;
        }

        public String getField() {
            return field;
        }

        public List<String> getJsonPath() {
            return jsonPath;
        }

        public Object getDefaultValue() {
            return defaultValue;
        }

        public String getDefaultExpr() {
            return defaultExpr;
        }
    }

    public static JsonParserConfig fromJson(JsonObject jsonParserObject, EngineType engineType) throws ConfigurationException {
        String dateFormatString = getFieldOrNull(jsonParserObject, "dateFormat");
        DateFormat dateFormat = dateFormatString == null ? null : new SimpleDateFormat(dateFormatString);

        JsonObject columnsObject = jsonParserObject.getAsJsonObject("columns");
        Preconditions.checkArgument(columnsObject != null);
        List<Column> columns = new ArrayList<>();
        for (Map.Entry<String, JsonElement> columnEntry : columnsObject.entrySet()) {
            columns.add(parseColumn(columnEntry, dateFormat));
        }
        return new JsonParserConfig(
            dateFormatString,
            getFieldOrNull(jsonParserObject, "timestampField"),
            getFieldOrNull(jsonParserObject, "timestampJsonPath"),
            getFieldOrNull(jsonParserObject, "dateField"),
            getFieldOrNull(jsonParserObject, "dateJsonPath"),
            columns,
            engineType
        );
    }

    private static Column parseColumn(Map.Entry<String, JsonElement> columnEntry,
                                      DateFormat dateFormat) throws ConfigurationException {
        String columnName = columnEntry.getKey();
        if (columnEntry.getValue().isJsonPrimitive()) {
            ColumnTypeBase type = ColumnTypeUtils.fromClickhouseDDL(columnEntry.getValue().getAsString());
            return new Column(columnName, type, columnName, null, null, null);
        }

        JsonObject columnParamsObject = columnEntry.getValue().getAsJsonObject();
        String field = getFieldOrNull(columnParamsObject, "field");
        String jsonPath = getFieldOrNull(columnParamsObject, "jsonPath");
        if (field == null && jsonPath == null) {
            field = columnName;
        }
        ColumnTypeBase type = ColumnTypeUtils.fromClickhouseDDL(columnParamsObject.get("type").getAsString());
        Object defaultValue = null;
        JsonElement defaultValueElement = columnParamsObject.get("defaultValue");
        if (defaultValueElement != null) {
            try {
                defaultValue = type.parseValue(columnParamsObject.get("defaultValue").getAsString(), dateFormat);
            } catch (Exception e) {
                throw new ConfigurationException(
                    String.format(
                        "Invalid default value for column %s(%s): %s", columnName, type, defaultValueElement.toString()
                    )
                );
            }
        }
        String defaultExpr = getFieldOrNull(columnParamsObject, "defaultExpr");
        return new Column(columnName, type, field, jsonPath, defaultValue, defaultExpr);
    }

    private static String getFieldOrNull(JsonObject jsonObject, String field) {
        JsonElement element = jsonObject.get(field);
        if (element == null) {
            return null;
        }
        String value = element.getAsString().trim();
        return value.isEmpty() ? null : value;
    }

    private static List<String> splitOptionalString(String path) {
        if (path == null) {
            return null;
        }
        return Collections.unmodifiableList(Arrays.asList(path.split("\\.")));
    }

}
