package ru.yandex.market.logshatter.logging;

import com.google.common.collect.HashMultiset;
import com.google.common.collect.Multiset;

import java.util.ArrayList;
import java.util.List;

/**
 * @author Anton Sukhonosenko <a href="mailto:algebraic@yandex-team.ru"></a>
 * @date 11.10.16
 */
public class BatchErrorLogger {
    private final int errorsCountToBeginGrouping;
    private final int errorsCountToBeginReducedGrouping;
    private final ErrorLogger logger;

    private List<ErrorSample> errors = new ArrayList<>();
    private Multiset<ErrorSample> errorMultiset = HashMultiset.create();

    private int nextBatchToLog = 1;
    private int totalErrorCount = 0;
    private int batchNumberAfterSecondThreshold = 1;

    /**
     * @param logger
     * @param errorsCountToBeginGrouping        Количество ошибок, при котором эксепшны начинают группироваться
     * @param errorsCountToBeginReducedGrouping Количество ошибок, при котором эксепшны логгируются без стэктрейсов
     */
    public BatchErrorLogger(ErrorLogger logger, int errorsCountToBeginGrouping, int errorsCountToBeginReducedGrouping) {
        this.logger = logger;
        this.errorsCountToBeginGrouping = errorsCountToBeginGrouping;
        this.errorsCountToBeginReducedGrouping = errorsCountToBeginReducedGrouping;
    }

    public void addError(Exception exception, String line) {
        ErrorSample sample = new ErrorSample(exception, line);

        if (totalErrorCount >= errorsCountToBeginGrouping) {
            errorMultiset.add(sample);
        } else {
            errors.add(sample);
        }

        totalErrorCount++;
    }

    public void batchParsed() {
        if (!errors.isEmpty()) {
            for (ErrorSample error : errors) {
                logger.logSingleError(error);
            }

            errors = new ArrayList<>();
        }

        if (totalErrorCount > errorsCountToBeginReducedGrouping) {
            if (batchNumberAfterSecondThreshold == nextBatchToLog) {
                for (ErrorSample sample : errorMultiset.elementSet()) {
                    int count = errorMultiset.count(sample);
                    logger.logErrorGroupWithoutStacktrace(sample, count);
                }

                errorMultiset.clear();

                // После второго порога сначала логгируем на 1-м батче, потом на 2, 4, 8, 16 и т.д.
                // Иными словами, пишем в лог всё реже и реже, пока лог, который мы парсим, не ротируется
                nextBatchToLog <<= 1;
            }

            batchNumberAfterSecondThreshold++;
        } else if (totalErrorCount > errorsCountToBeginGrouping) {
            for (ErrorSample sample : errorMultiset.elementSet()) {
                int count = errorMultiset.count(sample);
                logger.logErrorGroup(sample, count);
            }

            errorMultiset.clear();
        }
    }
}
