package ru.yandex.market.logshatter.meta;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.mongodb.MongoClient;
import com.mongodb.MongoClientOptions;
import com.mongodb.MongoClientURI;
import com.mongodb.ReadPreference;
import com.mongodb.WriteConcern;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Indexes;
import com.mongodb.client.model.UpdateManyModel;
import com.mongodb.client.model.UpdateOptions;
import com.mongodb.client.model.WriteModel;
import org.bson.BsonReader;
import org.bson.BsonWriter;
import org.bson.Document;
import org.bson.codecs.Codec;
import org.bson.codecs.DecoderContext;
import org.bson.codecs.DocumentCodec;
import org.bson.codecs.EncoderContext;
import org.bson.codecs.configuration.CodecRegistries;
import org.bson.codecs.configuration.CodecRegistry;
import ru.yandex.market.logshatter.reader.SourceContext;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/07/15
 */
public class LogshatterMetaDao {
    private static final UpdateOptions UPSERT = new UpdateOptions().upsert(true);

    private final MongoCollection<SourceMeta> collection;

    private static final CodecRegistry CODEC_REGISTRY = CodecRegistries.fromRegistries(
        MongoClient.getDefaultCodecRegistry(),
        CodecRegistries.fromCodecs(new SourceMetaCodec())
    );

    public LogshatterMetaDao(
        String mongoUrl,
        String dbName,
        int connectTimeoutMillis,
        int socketTimeoutMillis,
        String replicaSet,
        boolean ssl
    ) {
        this(createDatabase(mongoUrl, dbName, connectTimeoutMillis, socketTimeoutMillis, replicaSet, ssl));
    }

    @VisibleForTesting
    public LogshatterMetaDao(MongoDatabase database) {
        this.collection = database
            .withCodecRegistry(CODEC_REGISTRY)
            .getCollection("source", SourceMeta.class);
        collection.createIndex(Indexes.ascending("origin", "updated"));
    }

    private static MongoDatabase createDatabase(
        String mongoUrl,
        String dbName,
        int connectTimeoutMillis,
        int socketTimeoutMillis,
        String replicaSet,
        boolean ssl
    ) {
        MongoClientOptions.Builder options = MongoClientOptions.builder()
            .writeConcern(WriteConcern.MAJORITY)
            .readPreference(ReadPreference.primary())
            .connectTimeout(connectTimeoutMillis)
            .socketTimeout(socketTimeoutMillis)
            .sslEnabled(ssl);
        if (!Strings.isNullOrEmpty(replicaSet)) {
            options.requiredReplicaSetName(replicaSet);
        }
        return new MongoClient(new MongoClientURI(mongoUrl, options)).getDatabase(dbName);
    }

    private String toId(SourceKey sourceKey) {
        return sourceKey.getOrigin() + "-" + sourceKey.getId() + "-" + sourceKey.getTable();
    }

    private Document toIdDocument(SourceContext sourceContext) {
        return new Document("_id", toId(sourceContext.getSourceKey()));
    }

    public void save(SourceContext sourceContext) {
        save(Collections.singletonList(sourceContext));
    }

    @VisibleForTesting
    void save(SourceContext sourceContext, Date updateDate) {
        save(Collections.singletonList(sourceContext), updateDate);
    }

    public void save(Collection<SourceContext> sourceContexts) {
        save(sourceContexts, new Date());
    }

    private void save(Collection<SourceContext> sourceContexts, Date updateDate) {
        List<WriteModel<SourceMeta>> writeModels = new ArrayList<>();
        for (SourceContext sourceContext : sourceContexts) {
            writeModels.add(new UpdateManyModel<>(
                toIdDocument(sourceContext),

                toUpdateObject(sourceContext, updateDate),
                UPSERT
            ));
        }
        collection.bulkWrite(writeModels);
    }

    public void load(SourceContext sourceContext) {
        SourceMeta meta = get(sourceContext);
        if (meta != null) {
            sourceContext.setDataOffset(meta.getDataOffset());
            sourceContext.setFileOffset(meta.getFileOffset());
        }
    }

    public SourceMeta get(SourceContext sourceContext) {
        for (SourceMeta meta : collection.find(toIdDocument(sourceContext))) {
            return meta;
        }
        return null;
    }

    public Map<SourceKey, SourceMeta> getByOrigin(String origin) {
        Map<SourceKey, SourceMeta> sourceMetas = new HashMap<>();
        for (SourceMeta meta : collection.find(new Document("origin", origin))) {
            sourceMetas.put(meta.getKey(), meta);
        }
        return sourceMetas;
    }

    public int cleanupOldSources(String origin, Date beforeDate) {
        Document filter = new Document();
        filter.put("origin", origin);
        filter.put("updated", new Document("$lt", beforeDate));
        return (int) collection.deleteMany(filter).getDeletedCount();
    }

    private static class SourceMetaCodec implements Codec<SourceMeta> {
        private static final Codec<Document> DOCUMENT_CODEC = new DocumentCodec();

        @Override
        public SourceMeta decode(BsonReader reader, DecoderContext decoderContext) {
            Document document = DOCUMENT_CODEC.decode(reader, decoderContext);
            SourceKey key = new SourceKey(
                document.getString("origin"),
                document.getString("id"),
                document.getString("table")
            );
            return new SourceMeta(
                key,
                document.getString("name"),
                document.getLong("dataOffset"),
                document.getLong("fileOffset")
            );
        }

        @Override
        public void encode(BsonWriter writer, SourceMeta value, EncoderContext encoderContext) {
        }

        @Override
        public Class<SourceMeta> getEncoderClass() {
            return SourceMeta.class;
        }
    }

    private Document toUpdateObject(SourceContext sourceContext, Date updateDate) {
        SourceKey sourceKey = sourceContext.getSourceKey();
        Document document = new Document();
        document.put("origin", sourceKey.getOrigin());
        document.put("id", sourceKey.getId());
        document.put("table", sourceKey.getTable());
        document.put("dataOffset", sourceContext.getDataOffset());
        document.put("fileOffset", sourceContext.getFileOffset());
        document.put("name", sourceContext.getName());
        document.put("host", sourceContext.getHost());
        document.put("config", sourceContext.getLogShatterConfig().getShortConfigName());
        document.put("parser", sourceContext.getLogShatterConfig().getParserClassName());
        document.put("updated", updateDate);
        return new Document("$set", document);
    }

    public void delete(SourceContext sourceContext) {
        collection.deleteOne(toIdDocument(sourceContext));
    }

    @VisibleForTesting
    MongoCollection<SourceMeta> getCollection() {
        return collection;
    }
}
