package ru.yandex.market.logshatter.output;

import com.google.common.annotations.VisibleForTesting;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.reader.SourceContext;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 16/02/15
 */
public class OutputQueue {
    private static final Logger log = LogManager.getLogger();

    private final ReentrantLock lock = new ReentrantLock();
    private final Condition notEmpty;

    private final Set<ConfigOutputQueue> locked = new HashSet<>();
    private final Deque<ConfigOutputQueue> queue = new ArrayDeque<>();
    private final Set<ConfigOutputQueue> queuedItemsSetForDeduplication = new HashSet<>();

    private final Map<LogShatterConfig, ConfigOutputQueue> configQueues = new ConcurrentHashMap<>();


    public OutputQueue() {
        this(outputQueue -> outputQueue.lock.newCondition());
    }

    @VisibleForTesting
    OutputQueue(Function<OutputQueue, Condition> conditionFactory) {
        notEmpty = conditionFactory.apply(this);
    }


    public void add(SourceContext sourceContext) {
        lock.lock();

        ConfigOutputQueue configQueue = configQueues.get(sourceContext.getLogShatterConfig());
        if (configQueue == null) {
            configQueue = new ConfigOutputQueue(sourceContext.getLogShatterConfig());
            configQueues.put(sourceContext.getLogShatterConfig(), configQueue);
        }
        configQueue.add(sourceContext);

        if (!locked.contains(configQueue) && !queuedItemsSetForDeduplication.contains(configQueue)) {
            queue.addLast(configQueue);
            queuedItemsSetForDeduplication.add(configQueue);
            notEmpty.signal();
        }
        lock.unlock();
    }

    public void returnLockAndMaybeAddFirst(ConfigOutputQueue configQueue) {
        lock.lock();
        locked.remove(configQueue);
        recheck(configQueue, queue::addFirst);
        lock.unlock();
    }

    public void returnLockAndMaybeAddLast(ConfigOutputQueue configQueue) {
        lock.lock();
        locked.remove(configQueue);
        recheck(configQueue, queue::addLast);
        lock.unlock();
    }

    private void recheck(ConfigOutputQueue configQueue, Consumer<ConfigOutputQueue> adder) {
        lock.lock();
        if (!configQueue.isEmpty() && !queuedItemsSetForDeduplication.contains(configQueue)) {
            // Если сохраняли данные из ConfigOutputQueue и не смогли сохранить сразу всё, то добавляем эту
            // ConfigOutputQueue в начало очереди. Смотри https://st.yandex-team.ru/MARKETINFRA-4688.
            adder.accept(configQueue);
            queuedItemsSetForDeduplication.add(configQueue);
            notEmpty.signal();
        }
        lock.unlock();
    }

    public ConfigOutputQueue takeLock() throws InterruptedException {
        ConfigOutputQueue queue;
        while (true) {
            queue = doTakeLock();
            if (!queue.isEmpty()) {
                return queue;
            } else {
                returnLockAndMaybeAddLast(queue);
            }
        }
    }

    public ConfigOutputQueue doTakeLock() throws InterruptedException {
        lock.lock();
        while (queue.isEmpty()) {
            notEmpty.await();
        }
        ConfigOutputQueue configOutputQueue = queue.removeFirst();
        queuedItemsSetForDeduplication.remove(configOutputQueue);
        locked.add(configOutputQueue);
        lock.unlock();
        return configOutputQueue;
    }

    @Override
    public String toString() {
        lock.lock();
        int queueSize = 0;
        int lockedSize = 0;
        for (ConfigOutputQueue configOutputQueue : configQueues.values()) {
            queueSize += configOutputQueue.queueSize();
            lockedSize += configOutputQueue.lockedSize();
        }
        String string = "{queueSize=" + queueSize + ", processingSize=" + lockedSize + "}";
        lock.unlock();
        return string;

    }

    @VisibleForTesting
    boolean isEmpty() {
        return queue.isEmpty();
    }
}

