package ru.yandex.market.logshatter.parser;

import org.apache.commons.lang3.StringUtils;
import com.google.common.primitives.UnsignedLong;

import java.text.DateFormat;
import java.text.ParseException;
import java.util.Date;
import java.util.Map;

public class KeyValueExtractor {

    private final Map<String, String> values;

    public KeyValueExtractor(Map<String, String> values) {
        this.values = values;
    }

    public Integer getOptionalInt(String key, Integer defaultValue) {
        return ParseUtils.parseInt(values.get(key), defaultValue);
    }

    public Integer getInt(String key) throws ParserException {
        String value = getString(key);
        try {
            return Integer.valueOf(value);
        } catch (NumberFormatException e) {
            throw new ParserException("Failed to parse int value ('" + value + "') in line", e);
        }
    }

    public Double getOptionalDouble(String key, Double defaultValue) {
        return ParseUtils.parseDouble(values.get(key), defaultValue);
    }

    public Double getDouble(String key) throws ParserException {
        String value = getString(key);
        try {
            return Double.valueOf(value);
        } catch (NumberFormatException e) {
            throw new ParserException("Failed to parse double value ('" + value + "') in line", e);
        }

    }

    public Long getOptionalLong(String key, Long defaultValue) {
        return ParseUtils.parseLong(values.get(key), defaultValue);
    }

    public Long getLong(String key) throws ParserException {
        String value = getString(key);
        try {
            return Long.valueOf(value);
        } catch (NumberFormatException e) {
            throw new ParserException("Failed to parse long value ('" + value + "') in line", e);
        }
    }

    public UnsignedLong getOptionalUnsignedLong(String key, UnsignedLong defaultValue) {
        return ParseUtils.parseUnsignedLong(values.get(key), defaultValue);
    }

    public UnsignedLong getUnsignedLong(String key) throws ParserException {
        String value = getString(key);
        try {
            return UnsignedLong.valueOf(value);
        } catch (NumberFormatException e) {
            throw new ParserException("Failed to parse long value ('" + value + "') in line", e);
        }
    }

    public String getOptionalString(String key, String valueOnNull) {
        String value = values.get(key);
        return value != null ? value : valueOnNull;

    }

    @SuppressWarnings("unchecked")
    public <T extends Enum<T>> T getOptionalEnum(String key, T valueOnNull) {
        String value = getOptionalString(key, null);
        return value == null ? valueOnNull : T.valueOf((Class<T>) valueOnNull.getClass(), value);
    }

    public String getString(String key) throws ParserException {
        checkRequired(key);
        return values.get(key);
    }


    public Boolean getBoolean(String key) throws ParserException {
        String string = getString(key);
        if (string.isEmpty()) {
            return false;
        }
        if (string.equals("0") || string.equalsIgnoreCase("false")) {
            return false;
        }
        return true;
    }

    public Boolean getBooleanStrict(String key) throws ParserException {
        String string = getString(key);
        if (string.equals("0") || string.equalsIgnoreCase("false")) {
            return false;
        } else if (string.equals("1") || string.equalsIgnoreCase("true")) {
            return true;
        } else {
            throw new ParserException("Failed to parse boolean value ('" + string + "') in line");
        }
    }

    public Boolean getOptionalBoolean(String key, Boolean defaultValue) throws ParserException {
        String string = values.get(key);
        if (string == null || string.isEmpty()) {
            return defaultValue;
        }
        return getBooleanStrict(key);
    }

    public Date getDate(String key, DateFormat dateFormat) throws ParserException {
        String value = getString(key);
        try {
            return dateFormat.parse(value);
        } catch (ParseException e) {
            throw new ParserException("Failed to parse date value ('" + value + "') in line", e);
        }
    }

    /**
     * Defaults to the beginning of unix epoch.
     */
    public Date getOptionalDate(String key, DateFormat dateFormat) throws ParserException {
        String value = getString(key);
        if (StringUtils.isEmpty(key)) {
            return new Date(0L);
        }
        try {
            return dateFormat.parse(value);
        } catch (ParseException e) {
            throw new ParserException("Failed to parse date value ('" + value + "') in line", e);
        }
    }

    public String[] getStringArray(String key) throws ParserException {
        return getStringArray(key, ",");
    }

    public String[] getStringArray(String key, String separator) throws ParserException {
        checkRequired(key);
        String value = values.get(key);
        return !value.isEmpty() ? value.split(separator) : ParseUtils.EMPTY_STRING_ARRAY;
    }

    public Integer[] getIntArray(String key) throws ParserException {
        return ParseUtils.parseIntArray(getString(key));
    }

    public Integer[] getIntArray(String key, String separator) throws ParserException {
        return ParseUtils.parseIntArray(getString(key), separator);
    }

    public Integer[] getOptionalIntArray(String key, Integer[] valueOnNull) throws ParserException {
        return getOptionalIntArray(key, ",", valueOnNull);
    }

    public Integer[] getOptionalIntArray(String key, String separator, Integer[] valueOnNull) throws ParserException {
        String stringValue = getOptionalString(key, null);
        return stringValue == null ? valueOnNull : ParseUtils.parseIntArray(stringValue, separator);
    }

    public Long[] getLongArray(String key) throws ParserException {
        return ParseUtils.parseLongArray(getString(key));
    }

    public Long[] getLongArray(String key, String separator) throws ParserException {
        return ParseUtils.parseLongArray(getString(key), separator);
    }

    public Long[] getOptionalLongArray(String key, Long[] valueOnNull) throws ParserException {
        return getOptionalLongArray(key, ",", valueOnNull);
    }

    public Long[] getOptionalLongArray(String key, String separator, Long[] valueOnNull) throws ParserException {
        String stringValue = getOptionalString(key, null);
        return stringValue == null ? valueOnNull : ParseUtils.parseLongArray(stringValue, separator);
    }

    public Map<String, String> getValues() {
        return values;
    }

    private void checkRequired(String key) throws ParserException {
        if (key == null) {
            throw new NullPointerException("Key cannot be null");
        }
        if (!values.containsKey(key)) {
            throw new ParserException("Required value ('" + key + "') not found.");
        }
    }
}
