package ru.yandex.market.logshatter.parser;

import com.google.common.base.Preconditions;
import ru.yandex.market.logshatter.config.ConfigValidationException;
import ru.yandex.market.logshatter.config.ParserConfig;
import ru.yandex.market.logshatter.generic.json.JsonParser;
import ru.yandex.market.logshatter.generic.json.JsonParserConfig;
import ru.yandex.market.logshatter.parser.auto.AutoParser;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 01/11/2017
 */
public class LogParserProvider {

    private final Type type;
    private final Class<? extends LogParser> parserClass;
    private final ParserConfig parserConfig;
    private final JsonParserConfig jsonParserConfig;

    public LogParserProvider(String parserClassName, ParserConfig parserConfig,
                             JsonParserConfig jsonParserConfig) throws ConfigValidationException {
        this.type = getType(parserClassName, parserConfig, jsonParserConfig);
        this.parserClass = (parserClassName == null ? null : getClass(parserClassName));
        this.parserConfig = parserConfig;
        this.jsonParserConfig = jsonParserConfig;
    }

    public static Type getType(String parserClassName, ParserConfig parserConfig,
                               JsonParserConfig jsonParserConfig) throws ConfigValidationException {
        Type type = null;
        if (parserClassName != null) {
            type = Type.CLASS;
        }
        if (parserConfig != null) {
            Preconditions.checkArgument(type == null, "Only one parser type can pe provided");
            type = Type.AUTO;
        }
        if (jsonParserConfig != null) {
            Preconditions.checkArgument(type == null, "Only one parser type can pe provided");
            type = Type.JSON;
        }
        Preconditions.checkArgument(type != null, "No parsers configured");
        return type;
    }


    public LogParser createParser() {
        switch (type) {
            case CLASS:
                try {
                    return parserClass.newInstance();
                } catch (Exception e) {
                    throw new RuntimeException("Failed to create parser for class: " + parserClass.getName(), e);
                }
            case AUTO:
                return new AutoParser(parserConfig);
            case JSON:
                return new JsonParser(jsonParserConfig);
            default:
                throw new IllegalStateException();
        }
    }

    public String getName() {
        switch (type) {
            case CLASS:
                return parserClass.getName();
            case AUTO:
                return "AutoParser";
            case JSON:
                return "JsonParser";
            default:
                throw new IllegalStateException();
        }
    }

    public ParserConfig getParserConfig() {
        return parserConfig;
    }

    public String getShortName() {
        if (type == Type.CLASS) {
            return parserClass.getSimpleName();
        }
        return getName();
    }

    private static Class<? extends LogParser> getClass(String className) throws ConfigValidationException {
        Class clazz;
        try {
            clazz = Class.forName(className);
        } catch (ClassNotFoundException e) {
            throw new ConfigValidationException("Parser class not found: " + className, e);
        }

        if (!LogParser.class.isAssignableFrom(clazz)) {
            throw new ConfigValidationException("Parser class " + className + " is not instance of LogParser");
        }
        @SuppressWarnings("unchecked")
        Class<? extends LogParser> parserClass = (Class<? extends LogParser>) clazz;
        try {
            parserClass.newInstance();
        } catch (InstantiationException | IllegalAccessException e) {
            throw new ConfigValidationException("Failed to create instance of parser class: " + className, e);
        }
        return parserClass;
    }

    private enum Type {
        CLASS,
        AUTO,
        JSON
    }
}
