package ru.yandex.market.logshatter.parser;

import com.google.common.base.Splitter;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.common.net.InetAddresses;
import com.google.common.primitives.UnsignedLong;
import org.apache.commons.lang3.StringUtils;
import ru.yandex.common.util.date.DateUtil;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 28/07/15
 */
public class ParseUtils {

    public static final DateUtil.ThreadLocalDateFormat ISO_8601_DATE_FORMAT = new DateUtil.ThreadLocalDateFormat("yyyy-MM-dd'T'HH:mm:ssX");

    private static final Splitter TAB_SPLITTER = Splitter.on('\t').trimResults().omitEmptyStrings();
    private static final HashFunction SIP_HASH_64 = Hashing.sipHash24(0, 0);

    public static final Integer[] EMPTY_INTEGER_ARRAY = {};
    public static final Long[] EMPTY_LONG_ARRAY = {};
    public static final String[] EMPTY_STRING_ARRAY = {};

    private ParseUtils() {
    }

    public static Map<String, String> splitToKeyValue(String line) {
        Map<String, String> values = new HashMap<>();
        for (String kv : TAB_SPLITTER.split(line)) {
            String[] splits = kv.split("=", 2);
            if (splits.length != 2) {
                continue;
            }
            values.put(splits[0], splits[1]);
        }
        return values;
    }

    public static Integer parseBoolean(String string, Integer valueOnFail) {
        if (string == null || string.isEmpty()) {
            return valueOnFail;
        }
        return Boolean.valueOf(string) ? 1 : 0;
    }

    public static Integer parseInt(String string, Integer valueOnFail) {
        if (string == null || string.isEmpty()) {
            return valueOnFail;
        }
        try {
            return Integer.valueOf(string);
        } catch (NumberFormatException e) {
            return valueOnFail;
        }
    }

    public static Long parseLong(String string, Long valueOnFail) {
        if (string == null || string.isEmpty()) {
            return valueOnFail;
        }
        try {
            return Long.valueOf(string);
        } catch (NumberFormatException e) {
            return valueOnFail;
        }
    }

    public static UnsignedLong parseUnsignedLong(String string) {
        return parseUnsignedLong(string, UnsignedLong.valueOf(0));
    }

    public static UnsignedLong parseUnsignedLong(String string, UnsignedLong valueOnFail) {
        if (string == null || string.isEmpty()) {
            return valueOnFail;
        }
        try {
            return UnsignedLong.valueOf(string);
        } catch (NumberFormatException e) {
            return valueOnFail;
        }
    }

    public static Double parseDouble(String string, Double valueOnFail) {
        if (string == null || string.isEmpty()) {
            return valueOnFail;
        }
        try {
            return Double.valueOf(string);
        } catch (NumberFormatException e) {
            return valueOnFail;
        }
    }

    public static Integer parseUnsignedInt(String string, Integer valueOnFail) {
        int value = parseInt(string, valueOnFail);
        return value >= 0 ? value : valueOnFail;
    }

    public static Date parseDateInSeconds(String string) {
        int timeSeconds = Integer.parseInt(string);
        return new Date(TimeUnit.SECONDS.toMillis(timeSeconds));
    }

    public static List<String> parseStringList(String string, String separator) throws ParserException {
        return Splitter.on(separator).trimResults().omitEmptyStrings().splitToList(string);
    }

    public static Integer[] parseIntArray(String string, String separator) throws ParserException {
        List<String> strings = parseStringList(string, separator);
        if (strings.isEmpty()) {
            return EMPTY_INTEGER_ARRAY;
        }
        Integer[] array = new Integer[strings.size()];
        for (int i = 0; i < strings.size(); i++) {
            array[i] = Integer.valueOf(strings.get(i));
        }
        return array;
    }

    public static Integer[] parseIntArray(String string) throws ParserException {
        if (string == null || string.isEmpty()) {
            return EMPTY_INTEGER_ARRAY;
        }
        return parseIntArray(string, ",");
    }

    public static Long[] parseLongArray(String string, String separator) throws ParserException {
        List<String> strings = parseStringList(string, separator);
        if (strings.isEmpty()) {
            return EMPTY_LONG_ARRAY;
        }
        Long[] array = new Long[strings.size()];
        for (int i = 0; i < strings.size(); i++) {
            array[i] = Long.valueOf(strings.get(i));
        }
        return array;
    }

    public static Long[] parseLongArray(String string) throws ParserException {
        if (string == null || string.isEmpty()) {
            return EMPTY_LONG_ARRAY;
        }
        return parseLongArray(string, ",");
    }

    public static Date parseDateWithISO8601Fallback(String string, DateFormat dateFormat) throws ParseException {
        try {
            return dateFormat.parse(string);
        } catch (ParseException e) {
            return ISO_8601_DATE_FORMAT.parse(string);
        }
    }

    public static String deNull(String string) {
        return (string != null) ? string : "";
    }

    public static String extractMethod(String url) {
        url = cutQueryStringAndFragment(url);
        int positionAfterLastSlash = url.lastIndexOf('/') + 1;
        return url.substring(positionAfterLastSlash);
    }

    public static String cutQueryStringAndFragment(String url) {
        int paramsStart = url.indexOf("?");
        int fragmentStart = url.indexOf("#");

        int end = url.length();
        if (paramsStart >= 0) {
            end = Math.min(end, paramsStart);
        }
        if (fragmentStart >= 0) {
            end = Math.min(end, fragmentStart);
        }
        return url.substring(0, end);
    }

    public static String[] splitPatternToLevels(String pattern) {
        int colonIndex = pattern.indexOf(':');
        String method;
        String url;
        if (colonIndex >= 0) {
            method = pattern.substring(0, colonIndex);
            url = pattern.substring(colonIndex + 1);
        } else {
            method = "<all>";
            url = pattern;
        }
        url = StringUtils.substringBefore(url, "?");
        url = StringUtils.appendIfMissing(url, "/");
        url += method;
        return splitToLevels(url);
    }

    public static String[] splitToLevels(String url) {
        while (url.charAt(0) == '/' && url.length() > 1) {
            url = url.substring(1);
        }
        return url.split("/");
    }

    public static String extractStringParam(String url, String name) {
        int startIndex;
        int index = 0;

        while((startIndex = url.indexOf(name, index)) > 0) {
            if ((url.charAt(startIndex - 1) == '&' || url.charAt(startIndex - 1) == '?')
                && url.charAt(startIndex+name.length()) == '=') {
                break;
            }
            index = startIndex + name.length();
        }

        if(startIndex < 0) {
            return "";
        }

        int endIndex = url.indexOf('&', startIndex);
        if (endIndex < 0) {
            endIndex = url.length();
        }
        return url.substring(startIndex + name.length() + 1, endIndex);
    }

    public static String extractDecodedStringParam(String url, String name) {
        String result = ParseUtils.extractStringParam(url, name);
        try {
            return URLDecoder.decode(result, "UTF-8");
        } catch (UnsupportedEncodingException ignored) {
            return result;
        }
    }

    public static Integer extractIntegerParam(String url, String name) {
        String param = extractStringParam(url, name);
        if (param.isEmpty()) {
            return 0;
        }
        try {
            return Integer.valueOf(param);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

    public static Integer extractUnsignedIntegerParam(String url, String name) {
        Integer value = extractIntegerParam(url, name);
        return (value >= 0) ? value : 0;
    }

    public static String extractParamsSubstring(String url) {
        Objects.requireNonNull(url);

        int start = url.indexOf("?");
        if (start < 0) {
            return StringUtils.EMPTY;
        }

        int end = url.indexOf("#", start + 1);
        if (end >= 0) {
            return url.substring(start + 1, end);
        }
        return url.substring(start + 1);
    }

    public static UnsignedLong sipHash64(String string) {
        return UnsignedLong.fromLongBits(SIP_HASH_64.hashBytes(string.getBytes()).asLong());
    }

    // copy from https://www.mkyong.com/java/java-convert-ip-address-to-decimal-number/
    public static long ipv4ToLong(String ipAddress) {
        if (!InetAddresses.isInetAddress(ipAddress)) {
            return 0;
        }

        long result = 0;

        String[] ipAddressInArray = ipAddress.split("\\.");

        if (ipAddressInArray.length != 4) {
            return 0;
        }

        for (int i = 3; i >= 0; i--) {

            long ip = Long.parseLong(ipAddressInArray[3 - i]);

            //left shifting 24,16,8,0 and bitwise OR

            //1. 192 << 24
            //1. 168 << 16
            //1. 1   << 8
            //1. 2   << 0
            result |= ip << (i * 8);

        }

        return result;
    }

    public static Integer[] parseTestBuckets(String experimentsString) {
        if (experimentsString.isEmpty()) {
            return new Integer[0];
        }
        List<Integer> testId = new ArrayList<>();
        for (String s : experimentsString.split(";")) {
            try {
                testId.add(Integer.parseInt(s.split(",")[0]));
            } catch (NumberFormatException ignored) {
            }
        }
        return testId.toArray(new Integer[testId.size()]);
    }

    public static String[] parseStringArray(String value, String separator) throws ParserException {
        return !value.isEmpty() ? value.split(separator) : ParseUtils.EMPTY_STRING_ARRAY;
    }
}
