package ru.yandex.market.logshatter.parser;

import com.google.common.annotations.VisibleForTesting;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import ru.yandex.market.logshatter.LogBatch;
import ru.yandex.market.logshatter.config.LogShatterConfig;
import ru.yandex.market.logshatter.reader.SourceContext;
import ru.yandex.market.logshatter.url.PageMatcher;
import ru.yandex.market.logshatter.useragent.UserAgentDetector;

import java.nio.file.Path;
import java.util.Date;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 05/05/15
 */
public class ParserContextImpl implements ParserContext {

    private static final Logger parserLog = LogManager.getLogger("parser");

    private final LogBatch logBatch;
    private final SourceContext sourceContext;
    private final PageMatcher pageMatcher;
    private final boolean skipObsoleteData;
    private final float sampleRatio;
    private final UserAgentDetector userAgentDetector;

    public ParserContextImpl(LogBatch logBatch, SourceContext sourceContext, PageMatcher pageMatcher,
                             boolean skipObsoleteData, float sampleRatio, UserAgentDetector userAgentDetector) {
        this.logBatch = logBatch;
        this.sourceContext = sourceContext;
        this.pageMatcher = pageMatcher;
        this.skipObsoleteData = skipObsoleteData;
        this.sampleRatio = sampleRatio;
        this.userAgentDetector = userAgentDetector;
    }

    @VisibleForTesting
    static boolean isObsoleteData(long dateMillis, long currentMillis, LogShatterConfig logShatterConfig) {

        final int dataRotationDays = logShatterConfig.getDataRotationDays();
        if (dataRotationDays != 0) {
            final long obsoleteDateMillis = currentMillis - TimeUnit.DAYS.toMillis(dataRotationDays);
            if (dateMillis < obsoleteDateMillis) {
                return true;
            }
        }

        return false;
    }

    @Override
    public Path getFile() {
        return sourceContext.getPath();
    }

    @Override
    public String getHost() {
        return sourceContext.getHost().toLowerCase();
    }

    @Override
    public int getInstanceId() {
        return sourceContext.getInstanceId();
    }

    @Override
    public String getLogBrokerTopic() {
        return sourceContext.getLogBrokerTopic();
    }

    @Override
    public String getOrigin() {
        return sourceContext.getOrigin();
    }

    @Override
    public PageMatcher getPageMatcher() {
        return pageMatcher;
    }

    @Override
    public String getParam(String name) {
        return sourceContext.getParams().get(name);
    }

    @Override
    public Map<String, String> getParams() {
        return sourceContext.getParams();
    }

    @Override
    public UserAgentDetector getUserAgentDetector() {
        return userAgentDetector;
    }

    @Override
    public void write(Date date, Object... fields) {
        if (skipObsoleteData) {
            final LogShatterConfig logShatterConfig = sourceContext.getLogShatterConfig();

            if (isObsoleteData(date.getTime(), System.currentTimeMillis(), logShatterConfig)) {
                throw new ObsoleteDataException(
                    "Obsolete data received. Skip. Table: " + logShatterConfig.getTableName() + "; " +
                        "Date: " + date + "; " +
                        "Host: " + sourceContext.getHost() + "; " +
                        "Path: " + sourceContext.getPath() + "; " +
                        "SourceKey origin: " + sourceContext.getSourceKey().getOrigin() + "; " +
                        "SourceKey id: " + sourceContext.getSourceKey().getId()
                );
            }
        }

        if (sampleRatio == 1.0f || (sampleRatio != 0.0f && ThreadLocalRandom.current().nextFloat() < sampleRatio)) {
            logBatch.write(date, fields);
        }
    }
}
