package ru.yandex.market.logshatter.parser;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * @author Tatiana Litvinenko <a href="mailto:tanlit@yandex-team.ru"></a>
 * @date 25.05.2015
 */
public class TableDescription {
    public static final Column DATE_COLUMN = new Column("date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("timestamp", ColumnType.UInt32);
    public static final MergeTree DEFAULT_ENGINE = MergeTree.fromOldDefinition("date", "timestamp");

    private final List<Column> columns;
    private final EngineType engine;

    public TableDescription(List<Column> columns, EngineType engine) {
        this.columns = columns;
        this.engine = engine == null ? DEFAULT_ENGINE : engine;
        validateSameName();
    }

    private void validateSameName() {
        Set<String> names = new HashSet<>();
        for (Column column : columns) {
            if (!names.add(column.getName())) {
                throw new IllegalStateException("Two columns with similar name: " + column.getName());
            }
        }
    }

    public List<Column> getColumns() {
        return columns;
    }

    public EngineType getEngine() {
        return engine;
    }

    public static TableDescription createDefault(Column... columns) {
        return create(DEFAULT_ENGINE, Arrays.asList(columns));
    }

    public static TableDescription createDefault(List<Column> columns) {
        return create(DEFAULT_ENGINE, columns);
    }

    public static TableDescription createDefault(List<String> additionalPrimaryKeys, Column... columns) {
        List<String> primaryKey = new ArrayList<>();
        primaryKey.add(TIMESTAMP_COLUMN.getName());
        primaryKey.addAll(additionalPrimaryKeys);
        return create(MergeTree.fromOldDefinition(DATE_COLUMN.getName(), primaryKey), Arrays.asList(columns));
    }

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }

        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        TableDescription that = (TableDescription) o;

        if ((columns == null && that.columns != null) || (columns != null && that.columns == null)) {
            return false;
        }

        if (columns != null) {
            if (columns.size() != that.columns.size() || !new HashSet<>(columns).containsAll(that.columns)) {
                return false;
            }
        }

        return engine != null ? engine.equals(that.engine) : that.engine == null;
    }

    @Override
    public int hashCode() {
        int result = columns != null ? columns.hashCode() : 0;
        result = 31 * result + (engine != null ? engine.hashCode() : 0);
        return result;
    }
}
