package ru.yandex.market.logshatter.parser.checkout;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TskvSplitter;

import java.text.SimpleDateFormat;
import java.util.Date;

public class CarterCartLogParser implements LogParser {
    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
            new Column("host", ColumnType.String),
            new Column("event_type", EnumColumnType.enum8(CartLogActionType.class)),
            new Column("user_id", ColumnType.String),
            new Column("user_id_type", EnumColumnType.enum8(UserIdType.class)),
            new Column("list_id", ColumnType.UInt64),
            new Column("item_id", ColumnType.UInt64),
            new Column("item_type", EnumColumnType.enum8(CartItemType.class)),
            new Column("item_count", ColumnType.UInt32),
            new Column("ware_md5", ColumnType.String),
            new Column("model_id", ColumnType.String),
            new Column("hid", ColumnType.String),
            new Column("shop_id", ColumnType.String),
            new Column("price", ColumnType.Float64),
            new Column("fee_sum", ColumnType.Float64)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    /**
     * Описание полей
     * Дата (в формате "yyyy-MM-dd'T'HH:mm:ssZ")
     * host
     * event_type
     * user_id
     * user_id_type
     * list_id
     * item_id
     * item_type
     * item_count
     * ware_md5
     * model_id
     * hid
     * shop_id
     * price
     * fee_sum
     */
    @Override
    public void parse(String line, ParserContext context) throws Exception {
        if (!line.startsWith("tskv")) {
            return;
        }

        TskvSplitter splitter = new TskvSplitter(line);
        Date date = splitter.getDate("event_time", new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ"));
        String host = context.getHost();
        CartLogActionType eventType = CartLogActionType.valueOf(splitter.getString("event_type"));
        String userId = splitter.getString("user_id");
        UserIdType userIdType = UserIdType.valueOf(splitter.getString("user_id_type"));
        Long listId = splitter.getLong("list_id");
        Long itemId = splitter.getLong("item_id");
        CartItemType itemType = CartItemType.valueOf(splitter.getString("item_type"));
        Integer itemCount = splitter.getInt("item_count");
        String wareMd5 = splitter.getOptionalString("ware_md5", "");
        String modelId = splitter.getOptionalString("model_id", "");
        String hid = splitter.getOptionalString("hid", "");
        String shopId = splitter.getOptionalString("shop_id", "");
        Double price = splitter.getOptionalDouble("price", Double.NaN);
        Double feeSum = splitter.getOptionalDouble("fee_sum", Double.NaN);
        context.write(
                date,
                host,
                eventType,
                userId,
                userIdType,
                listId,
                itemId,
                itemType,
                itemCount,
                wareMd5,
                modelId,
                hid,
                shopId,
                price,
                feeSum
        );
    }

}
