package ru.yandex.market.logshatter.parser.checkout;

import com.google.common.base.MoreObjects;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.url.Page;
import ru.yandex.misc.lang.StringUtils;

import java.util.Collection;

public class CarterLogParser implements LogParser {
    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
            new Column("host", ColumnType.String),
            new Column("url", ColumnType.String),
            new Column("http_method", ColumnType.String),
            new Column("http_code", ColumnType.UInt16),
            new Column("resptime_ms", ColumnType.UInt32),
            new Column("method", ColumnType.String),
            new Column("uid", ColumnType.UInt64),
            new Column("dynamic", ColumnType.UInt8),
            new Column("market_request_id", ColumnType.String),
            new Column("kv_keys", ColumnType.ArrayString),
            new Column("kv_values", ColumnType.ArrayString)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    /**
     * Описание полей
     * Дата (в формате "dd/MMM/yyyy:HH:mm:ss Z")
     * Таймстемп в сек
     * client ip
     * http_method
     * geturl
     * http_status_code
     * execution_time_ms
     * user_id
     * dynamic (1/0)
     *
     * @param line
     * @param context
     * @throws Exception
     */
    @Override
    @SuppressWarnings("checkstyle:magicnumber")
    public void parse(String line, ParserContext context) throws Exception {
        String[] splits = line.split("\t");
        if (splits.length < 9) {
            return;
        }
        int timestampSeconds = Integer.parseInt(splits[1]);
        String httpMethod = splits[3];
        String url = splits[4];
        int statusCode = Integer.parseInt(splits[5]);
        int executionTimeMillis = Integer.parseInt(splits[6]);
        long uid = Long.parseLong(splits[7]);
        boolean isDynamic = splits[8].equals("1");
        String marketRequestId = splits.length > 9 ? splits[9] : "";
        String kvJson = splits.length > 12 ? splits[12] : null;

        KeyValueParserUtil.KeysAndValues keysAndValues = KeyValueParserUtil.parseKeysAndValues(kvJson);

        Page matchedPage = context.getPageMatcher().matchUrl("carter.market.yandex.net", httpMethod, url);

        // в данном случае нам не нужна часть ID, в которой содержится название HTTP метода
        String pageId = StringUtils.substringBefore(
            MoreObjects.firstNonNull(matchedPage, Page.EMPTY).getId(),
            "."
        );

        context.write(
                timestampSeconds, context.getHost(), url, httpMethod, statusCode, executionTimeMillis,
                pageId, uid, isDynamic, marketRequestId, keysAndValues.getKeys(), keysAndValues.getValues()
        );
    }

}
