package ru.yandex.market.logshatter.parser.checkout;

import com.google.gson.Gson;
import org.apache.commons.lang3.StringEscapeUtils;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TskvSplitter;

import java.util.Date;

/**
 * Описание: https://wiki.yandex-team.ru/market/marketplace/Dev/checkouter/logs/#checkout/market-checkouter-order
 * -event.log.
 *
 * @author kukabara
 */
public class CheckoutEventLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("order_id", ColumnType.UInt64),
        new Column("shop_id", ColumnType.UInt64),
        new Column("uid", ColumnType.UInt64),
        new Column("user_group", ColumnType.String),
        new Column("no_auth", ColumnType.UInt8),
        new Column("fake", ColumnType.UInt8),
        new Column("accept_method", ColumnType.String),
        new Column("user_id", ColumnType.String),
        new Column("counter", ColumnType.String),
        new Column("platform", ColumnType.String),
        new Column("device_type", ColumnType.String),
        new Column("browser", ColumnType.String),
        new Column("meta_info", ColumnType.String), // json
        new Column("email", ColumnType.String),
        new Column("phone", ColumnType.String),
        new Column("totalprice", ColumnType.UInt32),
        new Column("itemcount", ColumnType.UInt32),
        new Column("subsidy", ColumnType.UInt32),
        new Column("gps_lat", ColumnType.Float64),
        new Column("gps_lon", ColumnType.Float64)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        if (!line.startsWith("tskv")) {
            return;
        }
        TskvSplitter splitter = new TskvSplitter(line);

        long timestampSeconds = splitter.getLong("timestamp");
        long orderId = splitter.getLong("order_id");
        long shopId = splitter.getLong("shop_id");
        long uid = splitter.getLong("uid");
        String userGroup = splitter.getString("user_group");
        if (userGroup == null) {
            return;
        }
        boolean noAuth = splitter.getBoolean("no_auth");
        boolean fake = splitter.getBoolean("fake");
        String acceptMethod = splitter.getString("accept_method");
        if (acceptMethod == null) {
            return;
        }

        Gson gson = new Gson();
        String metaJson = StringEscapeUtils.unescapeJson(splitter.getOptionalString("meta", ""));
        OrderMeta meta = null;
        if (!metaJson.isEmpty() && !metaJson.equals("null")) {
            meta = gson.fromJson(metaJson, OrderMeta.class);
        }
        String email = splitter.getString("email");
        String phone = splitter.getString("phone");
        Integer totalprice = splitter.getDouble("totalprice").intValue();
        Integer itemcount = splitter.getInt("itemcount");
        Integer subsidy = splitter.getDouble("subsidy").intValue();
        Double gpsLat = splitter.getOptionalDouble("gps_lat", 0d);
        Double gpsLon = splitter.getOptionalDouble("gps_lon", 0d);

        context.write(new Date(timestampSeconds), context.getHost(),
            orderId, shopId, uid, userGroup, noAuth, fake, acceptMethod,
            meta != null && meta.userId != null ? meta.userId : "",
            meta != null && meta.counter != null ? meta.counter : "",
            meta != null && meta.platform != null ? meta.platform.toUpperCase() : "",
            meta != null && meta.deviceType != null ? meta.deviceType.toUpperCase() : "",
            meta != null && meta.browser != null ? meta.browser : "",
            metaJson, email, phone, totalprice, itemcount, subsidy, gpsLat, gpsLon
        );
    }

    static class OrderMeta {
        String userId;
        String counter; // touch
        String platform; // WindowsPhone, ANDROID, iOS
        String deviceType; // SMARTPHONE, TABLET
        String browser; // IEMobile, ChromeMobile, MobileSafari, UCBrowser, etc
    }
}
