package ru.yandex.market.logshatter.parser.checkout;

import com.google.common.base.MoreObjects;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.url.Page;
import ru.yandex.misc.lang.StringUtils;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * User: akolotnina
 * Date: 2015-03-10
 * Time: 20:02
 * 'Выкусывает' из адреса параметры (предполагается, что все числа в адресе это id, которые меняются от ручки к ручке).
 * также убирает из хвоста .json и последний /
 */
public class CheckoutLogParser implements LogParser {
    private static final Logger log = LoggerFactory.getLogger(CheckoutLogParser.class);

    private static final String HOST = "checkouter.market.http.yandex.net";

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
            new Column("host", ColumnType.String),
            new Column("url", ColumnType.String),
            new Column("http_method", ColumnType.String),
            new Column("http_code", ColumnType.UInt16),
            new Column("resptime_ms", ColumnType.Int32),
            new Column("market_request_id", ColumnType.String),
            new Column("external_time_ms", ColumnType.Int32),
            new Column("pure_time_ms", ColumnType.Int32),
            new Column("service_url", ColumnType.UInt8, "(url = '/ping') OR (url LIKE '/monitor%')"),
            new Column("raw_url", ColumnType.String),
            new Column("kv_keys", ColumnType.ArrayString),
            new Column("kv_values", ColumnType.ArrayString),
            new Column("login", ColumnType.String),
            new Column("vip", ColumnType.Int8),
            new Column("method", ColumnType.String),
            new Column("marketEnvironmentProfile", ColumnType.String, "'standard'")
    );

    private static final Map<String, String> UID_TO_LOGIN_MAPPING = createUidToLoginMapping();

    private static Map<String, String> createUidToLoginMapping() {
        LinkedHashMap<String, String> result = new LinkedHashMap<>();
        result.put("610316930", "SNMikulskiy");
        result.put("651297667", "NAAlymovaSberbank");
        result.put("651286085", "IVKondrashovSberbank");
        result.put("651262087", "AVMorozovSberbank");
        result.put("651279881", "PNVlasovSberbank");
        result.put("651251916", "SKKuznetsovSberbank");
        result.put("610315405", "SVKartashow");
        result.put("651273358", "AYLyakin");
        result.put("651281613", "AAVedyakhin");
        result.put("651257872", "BIZlatkis");
        result.put("651299480", "ASBelokopytov");
        result.put("610316037", "AINikolaew");
        result.put("651291159", "SAMaltsevSberbank");
        result.put("651295896", "SEOstrovsky");
        result.put("651275227", "IVMalakhovaSberbank");
        result.put("610314682", "AVZavarzin");
        result.put("610316318", "OIKomlyakov");
        result.put("610314213", "VGKreynin");
        result.put("651271248", "YGChupina");
        result.put("610313828", "TTShternlib");
        result.put("651264357", "AYTorbakhov");
        result.put("651259558", "AVBazarovSberbank");
        result.put("31248233", "abovsky");
        result.put("651305705", "IBKolomeyskiySberbank");
        result.put("651302877", "SVLebedSberbank");
        result.put("650640171", "topmanagerwatch");
        result.put("725902573", "ggogref");
        result.put("610313007", "LAKhasis");
        result.put("651270000", "olvganeev");
        result.put("49389489", "rvsn183");
        result.put("651282930", "AMMelkonyan");
        result.put("610317365", "AnVlPopow");
        result.put("41", "volozh");
        result.put("610313402", "MVPoletaev");
        result.put("651247228", "GOGrefSberbank");
        result.put("651308157", "TVZavyalovaSberbank");
        result.put("651277054", "AGKhlyzovSberbank");
        result.put("650579469", "markzavadsky");
        return result;
    }

    @SuppressWarnings("checkstyle:magicnumber")
    public void parse(String line, ParserContext context) throws Exception {
        String[] splits = line.split("\t");
        if (splits.length < 9) {
            return;
        }
        int timestampSeconds = Integer.parseInt(splits[1]);
        String httpMethod = splits[3];
        String url = splits[4];
        int statusCode = Integer.parseInt(splits[5]);
        int executionTimeMillis = Integer.parseInt(splits[6]);

        String marketRequestId = splits.length > 9 ? splits[9] : "";
        int externalTime = splits.length > 10 ? Integer.parseInt(splits[10]) : 0;
        int pureTime = splits.length > 11 ? Integer.parseInt(splits[11]) : 0;
        String kvJson = splits.length > 12 ? splits[12] : null;

        KeyValueParserUtil.KeysAndValues keysAndValues = KeyValueParserUtil.parseKeysAndValues(kvJson);
        int indexOfUidParam = Iterables.indexOf(keysAndValues.getKeys(), ("param.uid")::equals);
        String uid;
        if (indexOfUidParam != -1) {
            uid = Iterables.get(keysAndValues.getValues(), indexOfUidParam);
        } else {
            uid = "";
        }

        int indexOfMarketEnvParam = Iterables.indexOf(keysAndValues.getKeys(), ("market.environment.profile")::equals);
        String marketEnvironmentProfile;
        if (indexOfMarketEnvParam != -1) {
            marketEnvironmentProfile = Iterables.get(keysAndValues.getValues(), indexOfMarketEnvParam);
        } else {
            marketEnvironmentProfile = "";
        }

        Page matchedPage = context.getPageMatcher().matchUrl(HOST, httpMethod, url);
        String pageId = StringUtils.substringBefore(
            MoreObjects.firstNonNull(matchedPage, Page.EMPTY).getId(),
            "."
        );

        String cleanUrl = clean(url);
        boolean serviceUrl = isServiceUrl(cleanUrl);

        String login = UID_TO_LOGIN_MAPPING.getOrDefault(uid, "");
        context.write(
                timestampSeconds, context.getHost(),
                cleanUrl,
                httpMethod,
                statusCode,
                executionTimeMillis,
                marketRequestId,
                externalTime,
                pureTime,
                serviceUrl,
                url,
                keysAndValues.getKeys(),
                keysAndValues.getValues(),
                login,
                !login.isEmpty(),
                pageId,
                marketEnvironmentProfile
        );
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    private String clean(String url) {
        return url
            .replaceAll("/[0-9A-F]+\\.[0-9A-F]+", "")
            .replaceAll("/payments/[0-9a-f]+/markup", "/payments/markup")
            .replaceAll("/\\d+", "")
            .replaceAll("/$", "")
            .replaceAll("\\.json$", "");
    }

    private boolean isServiceUrl(String cleanedUrl) {
        return "/ping".equals(cleanedUrl) || cleanedUrl.startsWith("/monitor");
    }

}
