package ru.yandex.market.logshatter.parser.checkout;

import org.apache.commons.lang3.StringUtils;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.enums.EnumColumnType;
import ru.yandex.market.logshatter.parser.EnvironmentMapper;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.trace.Environment;
import ru.yandex.market.logshatter.parser.trace.TraceLogEntry;
import ru.yandex.market.request.trace.Module;

import java.util.Arrays;
import java.util.EnumSet;
import java.util.Set;
import java.util.stream.Collectors;

import static ru.yandex.market.logshatter.parser.trace.RequestType.OUT;
import static ru.yandex.market.request.trace.Module.CHECKOUTER;
import static ru.yandex.market.request.trace.Module.CHECKOUTER_PUSH_API;
import static ru.yandex.market.request.trace.Module.MARKETPLACE_STORAGE_ZOOKEEPER;
import static ru.yandex.market.request.trace.Module.PGAAS;

/**
 * Парсер trace-логов для формирования таблицы логирования исходящих запросов компонента, который пишет трассировку
 */
public class CheckouterOutgoingLogParser implements LogParser {
    private static final Set<Module> TARGET_MODULES_TO_EXCLUDE = EnumSet.of(
        PGAAS, MARKETPLACE_STORAGE_ZOOKEEPER, CHECKOUTER, CHECKOUTER_PUSH_API);
    private static final Set<String> TARGET_MODULE_NAMES_TO_EXCLUDE =
        TARGET_MODULES_TO_EXCLUDE.stream().map(Module::toString).collect(Collectors.toSet());

    private EnvironmentMapper environmentMapper = new EnvironmentMapper(EnvironmentMapper.LOGBROKER_PROTOCOL_PREFIX);

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList("target_module", "environment"),
        new Column("request_id", ColumnType.String),
        new Column("start_time_ms", ColumnType.UInt64),
        new Column("end_time_ms", ColumnType.UInt64),
        new Column("duration_ms", ColumnType.UInt32),
        new Column("host", ColumnType.String),
        new Column("target_module", ColumnType.String),
        new Column("target_host", ColumnType.String),
        new Column("environment", EnumColumnType.enum8(Environment.class)),
        new Column("request_method", ColumnType.String),
        new Column("http_code", ColumnType.Int16),
        new Column("retry_num", ColumnType.UInt8),
        new Column("error_code", ColumnType.String),
        new Column("protocol", ColumnType.String),
        new Column("http_method", ColumnType.String),
        new Column("query_params", ColumnType.String),
        new Column("yandex_uid", ColumnType.String),
        new Column("yandex_login", ColumnType.String),
        new Column("kv_keys", ColumnType.ArrayString),
        new Column("kv_values", ColumnType.ArrayString),
        new Column("test_ids", ColumnType.ArrayUInt32),
        new Column("response_size_bytes", ColumnType.Int32, "-1")
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        Environment environment = environmentMapper.getEnvironment(context);
        TraceLogEntry entry = new TraceLogEntry(line, context);

        if (!TARGET_MODULE_NAMES_TO_EXCLUDE.contains(entry.getTargetModule()) && entry.getRequestType() == OUT) {
            context.write(
                entry.getRequestId().getDate(),
                entry.getRequestId().toString(),
                entry.getStartTimeMs(),
                entry.getEndTimeMs(),
                entry.getDurationMs(),
                context.getHost(),
                entry.getTargetModule(),
                entry.getTargetHost(),
                environment,
                cleanUrl(entry.getRequestMethod()),
                entry.getHttpCode(),
                entry.getRetryNum(),
                entry.getErrorCode(),
                entry.getProtocol(),
                entry.getHttpMethod(),
                entry.getQueryParams(),
                entry.getYandexUid(),
                entry.getYandexLogin(),
                entry.getKvKeys().toArray(),
                entry.getKvValues().toArray(),
                entry.getTestIds(),
                entry.getResponseSizeBytes()
            );
        }
    }

    private String cleanUrl(String url) {
        if(StringUtils.isEmpty(url)){
            return url;
        }
        if(url.startsWith("/v1/fiscal_storages")){
            return "/v1/fiscal_storages";
        }

        return url
            .replaceAll("/[0-9A-F]+\\.[0-9A-F]+", "")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/markup", "/trust-payments/v2/payments/markup")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+","/trust-payments/v2/payments")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/orders/[0-9a-f]+/unhold","/trust-payments/v2/payments/orders/unhold")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/orders/[0-9a-f]+/resize","/trust-payments/v2/payments/orders/resize")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/clear","/trust-payments/v2/payments/clear")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/unhold","/trust-payments/v2/payments/unhold")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/start","/trust-payments/v2/payments/start")
            .replaceAll("/trust-payments/v2/refunds/[0-9a-f]+/start","/trust-payments/v2/refunds/start")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/deliver","/trust-payments/v2/payments/deliver")
            .replaceAll("/trust-payments/v2/payments/[0-9a-f]+/receipts/[0-9a-f]+","/trust-payments/v2/payments/receipts")
            .replaceAll("/api/v2/operations/[0-9a-f]+","/api/v2/operations")
            .replaceAll("/api/v2/operations/[0-9a-f]+/results_data","/api/v2/operations/results_data")
            .replaceAll("/track/[0-9]+/stop","/track/stop")
            .replaceAll("/order/[0-9]+","/order")
            .replaceAll("/preorder/[0-9]+","/preorder")
            .replaceAll("/\\d+", "")
            .replaceAll("/$", "")
            .replaceAll("\\.json$", "");
    }
}
