package ru.yandex.market.logshatter.parser.checkout.events;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.checkout.CheckoutJsonLogParser;

import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

/**
 * @author Nicolai Iusiumbeli <mailto:armor@yandex-team.ru>
 * date: 10/03/2017
 */
public class CheckoutOrderEventHistoryLogParser extends CheckoutJsonLogParser {
    @Override
    public TableDescription getTableDescription() {
        return TableDescription.createDefault(
            new Column("host", ColumnType.String),
            new Column("eventId", ColumnType.Int64),
            new Column("eventType", ColumnType.String),
            new Column("orderId", ColumnType.Int64),
            new Column("oldUid", ColumnType.Int64),
            new Column("newUid", ColumnType.Int64),
            new Column("oldMuid", ColumnType.Int64),
            new Column("newMuid", ColumnType.Int64),
            new Column("oldStatus", ColumnType.String),
            new Column("newStatus", ColumnType.String),
            new Column("oldDeliveryType", ColumnType.String),
            new Column("newDeliveryType", ColumnType.String),
            new Column("oldPaymentMethod", ColumnType.String),
            new Column("newPaymentMethod", ColumnType.String),
            new Column("context", ColumnType.String),
            new Column("global", ColumnType.UInt8),
            new Column("oldSubstatus", ColumnType.String),
            new Column("newSubstatus", ColumnType.String),
            new Column("shopId", ColumnType.Int64),
            new Column("regionId", ColumnType.Int64),
            new Column("rgb", ColumnType.String),
            new Column("preorder", ColumnType.UInt8),
            new Column("sku", ColumnType.String),
            new Column("uid", ColumnType.UInt64),
            new Column("email", ColumnType.String),
            new Column("phone", ColumnType.String),
            new Column("platform", ColumnType.String),
            new Column("fulfilment", ColumnType.UInt8),
            new Column("warehouseId", ColumnType.Int32),
            new Column("deliveryPartnerType", ColumnType.String),
            new Column("cargoTypes", ColumnType.ArrayUInt32)
        );
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonElement element = gson.fromJson(line, JsonElement.class);

        if (element == null || !element.isJsonObject()) {
            return;
        }

        JsonObject jsonObject = element.getAsJsonObject();

        if (jsonObject.has("events")) {
            EventsContainer eventsContainer = gson.fromJson(element, EventsContainer.class);

            for (Event event : eventsContainer.getEvents()) {
                writeEvent(event, context);
            }
        } else {
            Event event = gson.fromJson(element, Event.class);
            writeEvent(event, context);
        }
    }

    private void writeEvent(Event event, ParserContext context) {
        Order orderBefore = event.getOrderBefore();
        Order orderAfter = event.getOrderAfter();

        Delivery deliveryBefore = orderBefore.getDelivery();
        Delivery deliveryAfter = orderAfter.getDelivery();

        String platformBefore = orderBefore.getProperties() != null
            ? orderBefore.getProperties().getPlatform()
            : null;
        String platformAfter = orderAfter.getProperties() != null
            ? orderAfter.getProperties().getPlatform()
            : null;

        context.write(
            event.getTranDate(),
            event.getHost(),
            event.getId(),
            event.getType(),
            nvl(orderBefore.getId(), orderAfter.getId(), -1),
            orderBefore.getUid(),
            orderAfter.getUid(),
            orderBefore.getMuid(),
            orderAfter.getMuid(),
            orderBefore.getStatus(),
            orderAfter.getStatus(),

            deliveryBefore.getType(),
            deliveryAfter.getType(),

            orderBefore.getPaymentMethod(),
            orderAfter.getPaymentMethod(),
            nvl(orderBefore.getContext(), orderAfter.getContext(), ""),
            // на самом деле не может измениться, как и контекст
            nvl(orderBefore.getGlobal(), orderAfter.getGlobal(), false),
            orderBefore.getSubstatus(),
            orderAfter.getSubstatus(),
            orderAfter.getShopId(),
            orderAfter.getDelivery().getRegionId(),
            nvl(orderBefore.getRgb(), orderAfter.getRgb(), ""),
            nvl(orderBefore.getPreOrder(), orderAfter.getPreOrder(), false),
            Optional.ofNullable(orderAfter.getItems())
                .filter(oi -> oi.size() == 1)
                .map(oi -> oi.get(0))
                .map(OrderItem::getSku)
                .orElse(""),
            orderAfter.getBuyer().getUid(),
            orderAfter.getBuyer().getEmail(),
            orderAfter.getBuyer().getNormalizedPhone(),
            nvl(platformAfter, platformBefore, ""),
            nvl(orderBefore.getFulfilment(), orderAfter.getFulfilment(), false),
            orderAfter.getWarehouseId(),
            deliveryAfter.getDeliveryPartnerType(),
            orderAfter.getCargoTypes()
        );
    }

    private <T> T nvl(T... args) {
        return Stream.of(args)
            .filter(Objects::nonNull)
            .findFirst()
            .orElse(null);
    }
}
