package ru.yandex.market.logshatter.parser.common;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParseUtils;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.ParserException;
import ru.yandex.market.logshatter.parser.TableDescription;

import java.nio.file.Path;
import java.util.Arrays;
import java.util.Date;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"></a>
 * @date 13/11/15
 */
public class CommonKeyValueLogParser implements LogParser {


    private static final int SPLITS_COUNT_WO_SUBKEY = 3;
    private static final int SPLITS_COUNT_W_SUBKEY = 4;

    private static final String KEY_VALUE_POSTFIX = "-key-value.log";

    private volatile String module = null;


    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        Arrays.asList("module", "key", "subkey"),
        new Column("module", ColumnType.String),
        new Column("host", ColumnType.String),
        new Column("key", ColumnType.String),
        new Column("subkey", ColumnType.String),
        new Column("value", ColumnType.Float64)
    );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    private void checkModule(ParserContext context) {
        if (module != null) {
            return;
        }
        module = context.getParams().getOrDefault("module", null);
        if (module != null) {
            return;
        }
        Path path = context.getFile();
        module = path.getName(path.getNameCount() - 1).toString();
        if (module.endsWith(KEY_VALUE_POSTFIX)) {
            module = module.replace(KEY_VALUE_POSTFIX, "");
        }
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        checkModule(context);

        String[] splits = line.split("\t");
        if (splits.length != SPLITS_COUNT_WO_SUBKEY && splits.length != SPLITS_COUNT_W_SUBKEY) {
            throw new ParserException("Split size must be 3 (date, key, value) or for 4 (date, key, subkey, value) ");
        }
        Date date = ParseUtils.ISO_8601_DATE_FORMAT.parse(splits[0]);
        String key = splits[1];
        String subkey;
        Double value;
        if (splits.length == SPLITS_COUNT_W_SUBKEY) {
            subkey = splits[2];
            value = Double.valueOf(splits[3]);
        } else {
            subkey = "";
            value = Double.valueOf(splits[2]);
        }
        context.write(date, module, context.getHost(), key, subkey, value);
    }
}
