package ru.yandex.market.logshatter.parser.delivery.blue.market;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.checkout.CheckoutJsonLogParser;
import ru.yandex.market.logshatter.parser.checkout.events.Checkpoint;
import ru.yandex.market.logshatter.parser.checkout.events.Event;
import ru.yandex.market.logshatter.parser.checkout.events.Order;
import ru.yandex.market.logshatter.parser.checkout.events.Shipment;
import ru.yandex.market.logshatter.parser.checkout.events.Track;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toList;

public class CheckpointStatusMonitoringLogParser extends CheckoutJsonLogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("status", ColumnType.Int64),
        new Column("previousStatus", ColumnType.Int64),
        new Column("durationMillis", ColumnType.Int64),
        new Column("orderId", ColumnType.Int64),
        new Column("trackId", ColumnType.Int64),
        new Column("deliveryServiceId", ColumnType.Int64),
        new Column("color", ColumnType.String)
    );

    private static final String TRACK_CHECKPOINT_CHANGED = "TRACK_CHECKPOINT_CHANGED";

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonElement element = gson.fromJson(line, JsonElement.class);
        if (!element.isJsonObject()) {
            return;
        }
        JsonObject jsonObject = element.getAsJsonObject();
        Event event = gson.fromJson(jsonObject, Event.class);
        List<CheckpointStatusMonitoringRecord> records = processCheckpointStatuses(event);
        records.forEach(record -> writeEvent(record, context));
    }

    List<CheckpointStatusMonitoringRecord> processCheckpointStatuses(Event event) {
        if (TRACK_CHECKPOINT_CHANGED.equals(event.getType())) {
            List<Track> tracksAfter = getTracks(event.getOrderAfter());

            Map<Long, Track> tracksBeforeMap = getTracks(event.getOrderBefore()).stream()
                .collect(Collectors.toMap(Track::getId, Function.identity()));

            return tracksAfter.stream().flatMap(track -> {
                List<Long> existingCheckpointsIds = getOldCheckpointsIds(tracksBeforeMap, track);
                sortCheckpointsByDateAndCheckpointId(track);
                return getMonitoringRecords(event, track, existingCheckpointsIds).stream();
            }).collect(toList());
        }
        return Collections.emptyList();
    }

    private List<Long> getOldCheckpointsIds(Map<Long, Track> tracksBeforeMap, Track track) {
        return tracksBeforeMap.get(track.getId()).getCheckpoints().stream()
            .map(Checkpoint::getId)
            .collect(toList());
    }

    private void sortCheckpointsByDateAndCheckpointId(Track track) {
        track.getCheckpoints().sort(
            Comparator.comparing(Checkpoint::getDate)
                .thenComparingLong(Checkpoint::getTrackerCheckpointId)
        );
    }

    private List<CheckpointStatusMonitoringRecord> getMonitoringRecords(Event event, Track track,
                                                                        List<Long> oldCheckpointsIds) {
        List<CheckpointStatusMonitoringRecord> records = new ArrayList<>();
        Checkpoint previousCheckpoint = null;
        for (Checkpoint checkpoint : track.getCheckpoints()) {
            if (!oldCheckpointsIds.contains(checkpoint.getId())) {
                CheckpointStatusMonitoringRecord record = createRecord(event);
                record.setTrackId(track.getId());
                setStatusesAndDuration(record, previousCheckpoint, checkpoint);
                records.add(record);
            }
            previousCheckpoint = checkpoint;
        }
        return records;
    }

    private void setStatusesAndDuration(CheckpointStatusMonitoringRecord record,
                                        Checkpoint previousCheckpoint, Checkpoint checkpoint) {
        record.setStatus(checkpoint.getDeliveryStatus());
        if (previousCheckpoint != null) {
            record.setPreviousStatus(previousCheckpoint.getDeliveryStatus());
            long duration = checkpoint.getDate().getTime() - previousCheckpoint.getDate().getTime();
            record.setDurationMillis(duration);
        }
    }


    private List<Track> getTracks(Order order) {
        List<Shipment> parcels = order.getDelivery().getParcels();
        if (parcels == null) {
            return Collections.emptyList();
        }
        return parcels.stream()
            .flatMap(shipment -> Optional.ofNullable(shipment.getTracks())
                .map(Collection::stream)
                .orElseGet(Stream::empty))
            .collect(toList());
    }

    private CheckpointStatusMonitoringRecord createRecord(Event event) {
        CheckpointStatusMonitoringRecord record = new CheckpointStatusMonitoringRecord();
        record.setTranDate(event.getTranDate());
        record.setHost(event.getHost());
        record.setStatus(-1);
        record.setPreviousStatus(-1);
        record.setDurationMillis(-1L);
        record.setOrderId(event.getOrderAfter().getId());
        record.setTrackId(-1L);
        record.setDeliveryServiceId(event.getOrderAfter().getDelivery().getDeliveryServiceId());
        record.setColor(event.getOrderAfter().getRgb());
        return record;
    }

    private void writeEvent(CheckpointStatusMonitoringRecord record, ParserContext context) {
        context.write(
            record.getTranDate(),
            record.getHost(),
            record.getStatus(),
            record.getPreviousStatus(),
            record.getDurationMillis(),
            record.getOrderId(),
            record.getTrackId(),
            record.getDeliveryServiceId(),
            record.getColor()
        );
    }

}
