package ru.yandex.market.logshatter.parser.delivery.blue.market;

import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import ru.yandex.market.logshatter.parser.checkout.events.Event;
import ru.yandex.market.logshatter.parser.checkout.events.Order;
import ru.yandex.market.logshatter.parser.checkout.events.Shipment;
import ru.yandex.market.logshatter.parser.checkout.events.Track;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;
import static ru.yandex.market.logshatter.parser.delivery.blue.market.DeliveryOrderStatusEventType.TRACKCODE_ABSENCE;

public class DeliveryOrderStatusMonitoringHelper {

    private static final String PENDING = "PENDING";
    public static final String PROCESSING = "PROCESSING";
    static final String PARCEL_ERROR_STATUS = "ERROR";

    static OrderStatusMonitoringRecord createRecordWithShipmentId(Event event, Long shipmentId,
                                                                  DeliveryOrderStatusEventType status) {
        OrderStatusMonitoringRecord weightRecord = createRecord(event, status);
        weightRecord.setShipmentId(shipmentId);
        return weightRecord;
    }

    static List<OrderStatusMonitoringRecord> trackCodeAppearance(Event event) {
        Map<Long, Shipment> shipmentsById = event.getOrderAfter().getDelivery().getShipments().stream()
            .collect(Collectors.toMap(Shipment::getId, Function.identity()));
        Map<Long, List<Track>> trackCodesByShipment = groupTrackByShipment(shipmentsById);
        Map<Long, Shipment> oldShipmentsById = event.getOrderBefore().getDelivery().getShipments().stream()
            .collect(Collectors.toMap(Shipment::getId, Function.identity()));
        removeNotUpdatedTracks(trackCodesByShipment, oldShipmentsById);

        return trackCodesByShipment.entrySet().stream()
            .filter(en -> !CollectionUtils.isEmpty(en.getValue()))
            .map(en -> createTrackCodeAppearanceRecord(event, en.getKey(), en.getValue()))
            .flatMap(List::stream)
            .collect(Collectors.toList());
    }

    private static void removeNotUpdatedTracks(Map<Long, List<Track>> trackCodesByShipment, Map<Long, Shipment> oldShipmentsById) {
        trackCodesByShipment.forEach((newShipmentId, newTracks) -> {
            Shipment oldShipment = oldShipmentsById.get(newShipmentId);
            if (oldShipment != null && !CollectionUtils.isEmpty(oldShipment.getTracks())) {
                newTracks.removeIf(track -> {
                    Map<Long, String> oldTrackCodesById = oldShipment.getTracks().stream()
                        .collect(HashMap::new, (map, value) -> map.put(value.getId(), value.getTrackCode()),
                            HashMap::putAll);
                    return oldTrackCodesById.get(track.getId()) != null;
                });
            }
        });
    }

    private static Map<Long, List<Track>> groupTrackByShipment(Map<Long, Shipment> shipmentsById) {
        return shipmentsById.entrySet().stream()
            .filter(en -> !CollectionUtils.isEmpty(en.getValue().getTracks()))
            .collect(Collectors.toMap(Map.Entry::getKey,
                en -> en.getValue().getTracks().stream()
                    .filter(t -> !StringUtils.isEmpty(t.getTrackCode()))
                    .collect(Collectors.toList())));
    }

    static boolean pendingTransitionOccurs(Order orderAfter, Order orderBefore) {
        return PENDING.equals(orderAfter.getStatus()) && !PENDING.equals(orderBefore.getStatus())
            || PENDING.equals(orderBefore.getStatus()) && !PENDING.equals(orderAfter.getStatus());
    }


    static OrderStatusMonitoringRecord createRecord(Event event, DeliveryOrderStatusEventType eventType) {
        OrderStatusMonitoringRecord record = new OrderStatusMonitoringRecord();
        record.setTranDate(event.getTranDate());
        record.setHost(event.getHost());
        record.setOrderId(event.getOrderAfter().getId());
        record.setShipmentId(-1L);
        record.setDeliveryServiceId(event.getOrderAfter().getDelivery().getDeliveryServiceId());
        record.setEventType(eventType);
        record.setColor(event.getOrderAfter().getRgb());
        return record;
    }

    static List<OrderStatusMonitoringRecord> trackCodeAbsence(Event event) {
        return event.getOrderAfter().getDelivery().getShipments()
            .stream()
            .map(shipment -> createTrackCodeAbsenceRecord(event, shipment))
            .flatMap(List::stream)
            .collect(Collectors.toList());
    }

    private static List<OrderStatusMonitoringRecord> createTrackCodeAbsenceRecord(Event event, Shipment shipment) {
        if (CollectionUtils.isEmpty(shipment.getTracks())) {
            OrderStatusMonitoringRecord record = createRecord(event, TRACKCODE_ABSENCE);
            record.setDeliveryServiceId(event.getOrderAfter().getDelivery().getDeliveryServiceId());
            return Collections.singletonList(record);
        } else {
            return shipment.getTracks().stream()
                .filter(t -> t.getTrackCode() == null)
                .map(t -> {
                    OrderStatusMonitoringRecord record = createRecordWithShipmentId(event, shipment.getId(),
                        TRACKCODE_ABSENCE);
                    record.setDeliveryServiceId(t.getDeliveryServiceId());
                    return record;
                }).collect(toList());
        }
    }

    private static List<OrderStatusMonitoringRecord> createTrackCodeAppearanceRecord(Event event,
                                                                                     Long shipmentId,
                                                                                     List<Track> newTracks) {
        return newTracks.stream().map(t -> {
            OrderStatusMonitoringRecord record = createRecordWithShipmentId(event, shipmentId, TRACKCODE_ABSENCE);
            record.setDeliveryServiceId(t.getDeliveryServiceId());
            return record;
        }).collect(Collectors.toList());

    }
}
