package ru.yandex.market.logshatter.parser.delivery.blue.market;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.checkout.CheckoutJsonLogParser;
import ru.yandex.market.logshatter.parser.checkout.events.Delivery;
import ru.yandex.market.logshatter.parser.checkout.events.Event;
import ru.yandex.market.logshatter.parser.checkout.events.Order;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import static java.util.stream.Collectors.toList;
import static ru.yandex.market.logshatter.parser.delivery.blue.market.DeliveryOrderStatusEventType.PARCEL_ERROR;
import static ru.yandex.market.logshatter.parser.delivery.blue.market.DeliveryOrderStatusEventType.PENDING_TRANSITION;
import static ru.yandex.market.logshatter.parser.delivery.blue.market.DeliveryOrderStatusMonitoringHelper.*;

public class DeliveryOrderStatusMonitoringLogParser extends CheckoutJsonLogParser {

    private static final TableDescription TABLE_DESCRIPTION = TableDescription.createDefault(
        new Column("host", ColumnType.String),
        new Column("orderId", ColumnType.Int64),
        new Column("shipmentId", ColumnType.Int64),
        new Column("deliveryServiceId", ColumnType.Int64),
        new Column("eventType", ColumnType.String),
        new Column("color", ColumnType.String)
    );

    private final List<Function<Event, List<OrderStatusMonitoringRecord>>> processors =
        Arrays.asList(
            this::parcelError,
            this::pendingTransition,
            this::trackCodeAbsence,
            this::trackCodeAppearance
        );

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonElement element = gson.fromJson(line, JsonElement.class);
        if (!element.isJsonObject()) {
            return;
        }
        JsonObject jsonObject = element.getAsJsonObject();
        Event event = gson.fromJson(jsonObject, Event.class);
        List<OrderStatusMonitoringRecord> records = createMonitoringRecords(event);
        records.forEach(record -> writeEvent(record, context));
    }

    List<OrderStatusMonitoringRecord> createMonitoringRecords(Event event) {
        return processors.stream().map(f -> f.apply(event))
            .flatMap(List::stream)
            .collect(toList());
    }

    private List<OrderStatusMonitoringRecord> parcelError(Event event) {
        Order orderAfter = event.getOrderAfter();
        Delivery delivery = orderAfter.getDelivery();
        return delivery.getShipments()
            .stream().filter(s -> PARCEL_ERROR_STATUS.equals(s.getStatus()))
            .map(s -> createRecordWithShipmentId(event, s.getId(), PARCEL_ERROR))
            .collect(toList());
    }

    private List<OrderStatusMonitoringRecord> pendingTransition(Event event) {
        if (pendingTransitionOccurs(event.getOrderAfter(), event.getOrderBefore())) {
            OrderStatusMonitoringRecord record = createRecord(event, PENDING_TRANSITION);
            return Collections.singletonList(record);
        } else {
            return Collections.emptyList();
        }
    }

    private List<OrderStatusMonitoringRecord> trackCodeAbsence(Event event) {
        if (PROCESSING.equals(event.getOrderAfter().getStatus())) {
            return DeliveryOrderStatusMonitoringHelper.trackCodeAbsence(event);
        } else {
            return Collections.emptyList();
        }
    }

    private List<OrderStatusMonitoringRecord> trackCodeAppearance(Event event) {
        if (PROCESSING.equals(event.getOrderAfter().getStatus())) {
            return DeliveryOrderStatusMonitoringHelper.trackCodeAppearance(event);
        } else {
            return Collections.emptyList();
        }
    }

    private void writeEvent(OrderStatusMonitoringRecord record, ParserContext context) {
        context.write(
            record.getTranDate(),
            record.getHost(),
            record.getOrderId(),
            record.getShipmentId(),
            record.getDeliveryServiceId(),
            record.getEventType().toString(),
            record.getColor()
        );
    }

}
