package ru.yandex.market.logshatter.parser.direct;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nonnull;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.logshatter.parser.direct.logformat.DirectAccessFormat;
import ru.yandex.market.logshatter.parser.direct.logformat.DirectAccessFormatList;

import static java.util.Collections.emptyMap;

public class AccessLogParser implements LogParser {
    private static final Logger logger = LoggerFactory.getLogger(AccessLogParser.class);


    public static final String DATE_PATTERN = "'['dd/MMM/yyyy:HH:mm:ss Z']'";
    public final SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(body_bytes_sent)";

    public static List<String> additionalPrimaryKeys = Arrays.asList("vhost", semplName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE = new MergeTree("toYYYYMM(" + partName + ")", primaryKeys,
        semplName, 8192);

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("remote_addr", ColumnType.String),
        new Column("upstream_http_x_accel_uid", ColumnType.String),
        new Column("request_type", ColumnType.String),
        new Column("request_url", ColumnType.String),
        new Column("request_potocol", ColumnType.String),
        new Column("status", ColumnType.Int32),
        new Column("body_bytes_sent", ColumnType.Int32),
        new Column("http_referer", ColumnType.String),
        new Column("http_user_agent", ColumnType.String),
        new Column("http_host", ColumnType.String),
        new Column("server_port", ColumnType.Int16),
        new Column("upstream_response_time", ColumnType.ArrayFloat32),
        new Column("request_time", ColumnType.Float32),
        new Column("upstream_reqid", ColumnType.Int64),
        new Column("upstream_service", ColumnType.String),
        new Column("upstream_method", ColumnType.String),
        new Column("upstream_var.name", ColumnType.ArrayString),
        new Column("upstream_var.value", ColumnType.ArrayString),
        new Column("ssl_protocol", ColumnType.String),
        new Column("ssl_cipher", ColumnType.String),
        new Column("vhost", ColumnType.String),
        new Column("hostname", ColumnType.String)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {

        DirectAccessFormatList matcher = new DirectAccessFormat(line).parseList("\t");
        if (matcher.size() < 3) {
            return;
        }

        String hostname = context.getHost();
        String remote_addr = matcher.get(0).parseString();
        String upstream_http_x_accel_uid = matcher.get(1).parseString();
        Date date = dateFormat.parse(matcher.get(2).parseString());
        DirectAccessFormatList request = matcher.get(3).parseList(" ");
        String request_type = request.sget(0).parseString();
        String request_url = request.sget(1).parseString();
        String request_potocol = request.sget(2).parseString();
        int status = matcher.get(4).parseInt();
        int body_bytes_sent = matcher.get(5).parseInt();
        String http_referer = matcher.get(6).parseString();
        String http_user_agent = matcher.get(7).parseString();
        DirectAccessFormatList server = matcher.get(8).parseList(",");
        String http_host = server.sget(0).parseString();
        int server_port = server.sget(1).parseInt();
        DirectAccessFormatList times = matcher.get(9).parseList(":");
        List<Float> upstream_response_time = new ArrayList<>();
        for (int i = 0; i < times.size() - 1; i++) {
            Float value = times.sget(i).parseFloat();
            upstream_response_time.add(value);
        }

        float request_time = times.sget(times.size() - 1).parseFloat();
        String upstream_http_x_accel_info = matcher.get(10).parseString();
        String ssl_protocol = matcher.get(11).parseString();
        String ssl_cipher = matcher.get(12).parseString();
        String vhost = matcher.sget(13).parseString();

        UpstreamAccelInfo accelInfo = UpstreamAccelInfo.parse(upstream_http_x_accel_info);

        context.write(date, remote_addr, upstream_http_x_accel_uid, request_type, request_url, request_potocol, status,
            body_bytes_sent, http_referer, http_user_agent, http_host, server_port,
            upstream_response_time, request_time,

            accelInfo.reqid, accelInfo.service, accelInfo.method,
            new ArrayList<>(accelInfo.vars.keySet()),
            new ArrayList<>(accelInfo.vars.values()),

            ssl_protocol, ssl_cipher, vhost, hostname);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    static class UpstreamAccelInfo {
        private static final UpstreamAccelInfo EMPTY = new UpstreamAccelInfo(0, "", "", emptyMap());

        final long reqid;
        final String service;
        final String method;
        final Map<String, String> vars;

        UpstreamAccelInfo(
            long reqid,
            @Nonnull String service,
            @Nonnull String method,
            @Nonnull Map<String, String> vars)
        {
            this.reqid = reqid;
            this.service = service;
            this.method = method;
            this.vars = vars;
        }

        static UpstreamAccelInfo parse(String val) {
            if (val == null || val.equals("")) {
                return EMPTY;
            }

            long reqid = 0;
            String service = "";
            String method = "";
            // we need order for tests
            Map<String, String> vars = new LinkedHashMap<>();

            for (String part : val.split(",")) {
                String[] data = part.split(":", 2);
                if (data.length == 0) {
                    continue;
                }
                String name = data[0];
                String value = data.length > 1 ? data[1] : "";
                switch (name) {
                    case "reqid":
                        reqid = safeLongParse(value);
                        break;
                    case "cmd":
                        String[] cmd = value.split("/", 2);
                        if (cmd.length == 1) {
                            method = cmd[0];
                        } else if (cmd.length == 2) {
                            service = cmd[0];
                            method = cmd[1];
                        }
                        break;
                    default:
                        vars.put(name, value);
                }
            }

            return new UpstreamAccelInfo(reqid, service, method, vars);
        }

        private static long safeLongParse(String value) {
            try {
                return Long.parseLong(value);
            } catch (NumberFormatException e) {
                logger.warn("Can't parse reqid: {}", value);
                return 0;
            }
        }
    }
}
