package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;

import java.lang.Long;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonElement;

public class BinlogLogParser implements LogParser {

    private static final TableDescription TABLE_DESCRIPTION = createTableDescription();
    private static TableDescription createTableDescription() {
        String dateColumnName = "date";
        String timestampColumnName = "datetime";
        List<Column> columns = new ArrayList<Column>(Arrays.asList(
            new Column(dateColumnName, ColumnType.Date),
            new Column(timestampColumnName, ColumnType.DateTime),
            new Column("reqid", ColumnType.Int64),
            new Column("method", ColumnType.String),
            new Column("service", ColumnType.String),
            new Column("source", ColumnType.String),
            new Column("db", ColumnType.String),
            new Column("table", ColumnType.String),
//            new Column("operation", EnumColumnType.enum8(Operation.class)),
            new Column("operation", ColumnType.Int8),
            new Column("gtid_src", ColumnType.String),
            new Column("gtid_scn", ColumnType.Int64),
            new Column("query_seq_num", ColumnType.UInt32),
            new Column("primary_key", ColumnType.String),
            new Column("primary_key_schema", ColumnType.String),
            new Column("row.name", ColumnType.ArrayString, "emptyArrayString()", "[]"),
            new Column("row.value", ColumnType.ArrayString, "emptyArrayString()", "[]"),
            new Column("row.is_null", ColumnType.ArrayInt8) // сейчас это enum
       ));
       List<String> primaryKeys = new ArrayList<String>(Arrays.asList(
            "date",
            "query_seq_num"
       ));

       MergeTree engineType = new MergeTree("toYYYYMM(" + dateColumnName + ")", primaryKeys, null, 8192);
       return new TableDescription(columns, engineType);
    }
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
    private static final Gson gson = new Gson();
    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject rec = gson.fromJson(line, JsonObject.class);
        long timestamp = rec.get("utcTimestamp").getAsLong() * 1000;
        Date date = new Date(timestamp);

        long reqid = rec.get("traceInfoReqId").getAsLong();
        String service = rec.get("traceInfoService").getAsString();
        String method = rec.get("traceInfoMethod").getAsString();
        String source = rec.get("source").getAsString();
        String gtidSrc = rec.get("serverUuid").getAsString();
        long gtidScn = rec.get("transactionId").getAsLong();
        int querySeqNum = rec.get("queryIndex").getAsInt();
        String gtid = gtidSrc + ':' + Long.toString(gtidScn);
        String db = rec.get("db").getAsString();
        String table = rec.get("table").getAsString();
        String operationStr = rec.get("operation").getAsString();
        int operation;
        switch (operationStr) {
            case "INSERT":
                operation = 1;
                break;
            case "UPDATE":
                operation = 2;
                break;
            case "DELETE":
                operation = 3;
                break;
            default:
                operation = 0;
                break;
        }
        for (JsonElement elem : rec.get("rows").getAsJsonArray()) {
            int changeSeqNum = elem.getAsJsonObject().get("rowIndex").getAsInt();
            Set<Map.Entry<String, JsonElement>> rowsEntrySet = elem.getAsJsonObject().getAsJsonObject("after").entrySet();
            Map<String, JsonElement> rowKV = new HashMap<String, JsonElement>();
            for (Map.Entry<String, JsonElement> kv : rowsEntrySet) {
                rowKV.put(kv.getKey(), kv.getValue());
            }
            List<String> rowNames = new ArrayList<>(rowKV.keySet());
            List<String> rowValues = new ArrayList<String>();
            List<Integer> rowIsNull = new ArrayList<Integer>();
            Collections.sort(rowNames, String.CASE_INSENSITIVE_ORDER); 
            for (String k : rowNames) {
                JsonElement valueObject = rowKV.get(k);
                String v;
                if (valueObject.isJsonPrimitive() && !valueObject.isJsonNull()) {
                    v = valueObject.getAsString();
                } else {
                    v = valueObject.toString();
                }
                rowValues.add(v);
                int isNull = valueObject.isJsonNull() ? 1 : 0;
                rowIsNull.add(isNull);
            }

            Set<Map.Entry<String, JsonElement>> pkEntrySet = elem.getAsJsonObject().getAsJsonObject("primaryKey").entrySet();
            Map<String, JsonElement> pkKV = new HashMap<String, JsonElement>();
            for (Map.Entry<String, JsonElement> kv : pkEntrySet) {
                pkKV.put(kv.getKey(), kv.getValue());
            }
            List<String> pkColNames = new ArrayList<>(pkKV.keySet());
            Collections.sort(pkColNames, String.CASE_INSENSITIVE_ORDER);
            StringBuilder pkBuilder = new StringBuilder("");
            StringBuilder pkSchemaBuilder = new StringBuilder("");
            for (String k : pkColNames) {
                JsonElement valueObject = pkKV.get(k);
                String v;
                if (valueObject.isJsonPrimitive() && !valueObject.isJsonNull()) {
                    v = valueObject.getAsString();
                } else {
                    v = valueObject.toString();
                }
                if (pkBuilder.length() != 0) {
                    pkBuilder.append(",");
                    pkSchemaBuilder.append(",");
                }
                pkSchemaBuilder.append(k);
                pkBuilder.append(v);
            }
            String primaryKey = pkBuilder.toString();
            String primaryKeySchema = pkSchemaBuilder.toString();
            context.write(date, reqid, method, service, source, db, table, operation, gtidSrc, gtidScn,
                querySeqNum, primaryKey, primaryKeySchema, rowNames, rowValues, rowIsNull);
        }

    }

}
