package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;

import java.lang.Long;

import java.util.Date;
import java.util.List;
import java.util.ArrayList;
import java.util.Arrays;

import com.google.gson.Gson;
import com.google.gson.JsonObject;

public class BinlogQueryLogParser implements LogParser {
    public static final int MAX_QUERY_LENGTH = 16384;

    private static final TableDescription TABLE_DESCRIPTION = createTableDescription();
    private static TableDescription createTableDescription() {
        String dateColumnName = "date";
        String timestampColumnName = "datetime";
        List<Column> columns = new ArrayList<Column>(Arrays.asList(
            new Column(dateColumnName, ColumnType.Date),
            new Column(timestampColumnName, ColumnType.DateTime),
            new Column("reqid", ColumnType.Int64),
            new Column("method", ColumnType.String),
            new Column("service", ColumnType.String),
            new Column("source", ColumnType.String),
            new Column("gtid", ColumnType.String),
            new Column("gtid_src", ColumnType.String),
            new Column("gtid_scn", ColumnType.Int64),
            new Column("query_seq_num", ColumnType.UInt32),
            new Column("query", ColumnType.String, null, null, "ZSTD(5)")
       ));
       List<String> primaryKeys = new ArrayList<String>(Arrays.asList(
            "date",
            "gtid",
            "query_seq_num"
       ));

       MergeTree engineType = new MergeTree("toYYYYMM(" + dateColumnName + ")", primaryKeys, null, 8192);
       return new TableDescription(columns, engineType);
    }
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
    private static final Gson gson = new Gson();
    
    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject rec = gson.fromJson(line, JsonObject.class);
        long timestamp = rec.get("timestamp").getAsLong() * 1000;
        Date date = new Date(timestamp);

        long reqid = rec.get("traceInfoReqId").getAsLong();
        String service = rec.get("traceInfoService").getAsString();
        String method = rec.get("traceInfoMethod").getAsString();
        String source = rec.get("source").getAsString();
        String gtidSrc = rec.get("serverUuid").getAsString();
        long gtidScn = rec.get("transactionId").getAsLong();
        int querySecNum = rec.get("queryIndex").getAsInt();
        String query = truncateQuery(rec.get("sql").getAsString());
        
        String gtid = "";

        context.write(date, reqid, method, service, source, gtid, gtidSrc, gtidScn, querySecNum, query);
    }

    private String truncateQuery(String query) {
        if (query == null) {
            return null;
        } else if (query.length() <= MAX_QUERY_LENGTH) {
            return query;
        } else {
            return "<truncated, actual_length=" + query.length() + "> " + query.substring(0, MAX_QUERY_LENGTH);
        }
    }
}
