package ru.yandex.market.logshatter.parser.direct;

import java.io.StringReader;

import org.json.JSONObject;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.*;

import com.google.gson.stream.JsonReader;
import com.google.gson.*;

import ru.yandex.market.logshatter.parser.direct.logformat.BsPricesFormat;

public class BsExportPricesLogParser implements LogParser {
    public static final String DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateTimeFormat = new SimpleDateFormat(DATE_TIME_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);
    public static final Column CID_COLUMN = new Column("cid", ColumnType.Int64);

    private static final String sampleName = "intHash64(cid)";

    public static List<String> additionalPrimaryKeys = Arrays.asList(sampleName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final String DATATYPE = "request";

    private static final MergeTree
        DEFAULT_ENGINE = new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, sampleName, 1024);

    private static List<Column> columnsStruct = new ArrayList<>(Arrays.asList(
        new Column("par_id", ColumnType.Int64),
        new Column("par_type", ColumnType.String),
        new Column("data_type", ColumnType.String),
        new Column("pid", ColumnType.Int64),
        new Column("source", ColumnType.String),
        new Column("metadata", ColumnType.String),
        new Column("data", ColumnType.String, null, null, "ZSTD(5)")
    ));

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(CID_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 3);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.add(CID_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    // за основу взята ./protected/BS/Export/Queues.pm
    private static final Map<String, Integer> parIdsType = new HashMap<String, Integer>() {{
        put("std", 0);          //"стандартная очередь"
        put("camp", 10);        //"очередь заказов и новых кошельков"
        put("stdprice", 19);    //"очередь цен"
        put("heavy", 29);       //"тяжёлая очередь"
        put("fast", 40);        //"быстрая очередь"
        put("dev1", 50);        //"разработческая очередь 1"
        put("dev2", 55);        //"разработческая очередь 2"
        put("preprod", 74);     //"очередь preprod"
        put("devprice", 80);    //"разработческая очередь цен 1"
        put("devprice2", 85);   //"разработческая очередь цен 2"
        put("buggy", 90);       //"buggy очередь"
        put("std", 209);        //"очередь для полной переотправки в LB",
        put("notfound", -1);     //"не найдена информация об очереди"
    }};

    private Long getParIdsType(String value) {
        String[] queue = value.split("_");
        int groupId = parIdsType.containsKey(queue[0]) ? parIdsType.get(queue[0]) : -1;
        int personId = (queue.length > 1) ? Integer.parseInt(queue[(queue.length - 1)]) : 0;
        return new Long(groupId + personId);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {

        HashMap<BsPricesFormat, HashMap<Long, List<String>>> values = new HashMap<>();
        JsonReader reader;
        BsPricesFormat prices;

        try {
            reader = new JsonReader(new StringReader(line));
            Gson gson = new GsonBuilder().create();

            reader.beginArray();

            while (reader.hasNext()) {
                prices = gson.fromJson(reader, BsPricesFormat.class);

                prices.setDataType(DATATYPE);
                if (!prices.getLevel().equals("PRICE")) {
                    continue;
                }

                if (prices.getFullExportFlag()) {
                    continue;
                }

                if (values.get(prices) == null) {
                    values.put(prices, new HashMap<>());
                }

                JsonObject jData = gson.fromJson(prices.getData(), JsonObject.class);
                Long id = jData.get("BannerGroupEID").getAsLong();
                if (values.get(prices).get(id) == null) {
                    values.get(prices).put(id, new ArrayList<>());
                }
                values.get(prices).get(id).add(jData.toString());
            }

            reader.endArray();
            reader.close();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage());
        }

        for (BsPricesFormat key : values.keySet()) {
            Date dateTime = dateTimeFormat.parse(key.getTime());
            for (Long id : values.get(key).keySet()) {
                context.write(dateTime,
                    key.getCid(),
                    getParIdsType(key.getParType()),
                    key.getParType(),
                    key.getDataType(),
                    key.getPid(),
                    "file:" + context.getHost() + context.getFile().toString(),
                    key.toString(),
                    values.get(key).get(id).toString()
                );
            }
        }
    }
}
