package ru.yandex.market.logshatter.parser.direct;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.stream.JsonReader;
import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import java.io.IOException;
import java.io.StringReader;
import java.text.SimpleDateFormat;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.*;


import ru.yandex.market.logshatter.parser.direct.logformat.BsimportActiveOrdersFormat;

public class BsimportActiveOrdersLogParser implements LogParser {

    public static final String DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateTimeFormat = new SimpleDateFormat(DATE_TIME_PATTERN);

    private static final DateTimeFormatter dtf = DateTimeFormatter.ofPattern(DATE_TIME_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);
    public static final Column OID_COLUMN = new Column("order_id", ColumnType.Int64);

    private static final String sampleName = "intHash64(order_id)";

    public static List<String> additionalPrimaryKeys = Arrays.asList(sampleName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree
        DEFAULT_ENGINE = new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, sampleName, 8192);

    private static List<Column> columnsStruct = new ArrayList<>(Arrays.asList(
        new Column("stop", ColumnType.UInt8),
        new Column("shows", ColumnType.Int64),
        new Column("clicks", ColumnType.Int64),
        new Column("units_spent", ColumnType.Int64),
        new Column("micro_cost", ColumnType.Int64),
        new Column("micro_cost_cur", ColumnType.Int64),
        new Column("update_time", ColumnType.DateTime)
    ));

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(OID_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 3);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.add(OID_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        List<BsimportActiveOrdersFormat> values = new ArrayList();
        JsonReader reader;
        Date logTime = dateTimeFormat.parse("1970-01-02 13:21:35");

        try {
            reader = new JsonReader(new StringReader(line));
            Gson gson = new GsonBuilder().create();

            reader.beginObject();

            while (reader.hasNext()) {
                String name = reader.nextName();
                if (name.equals("data")) {
                    reader.beginArray();
                    while (reader.hasNext()) {
                        BsimportActiveOrdersFormat bsimport = gson.fromJson(reader, BsimportActiveOrdersFormat.class);
                        values.add(bsimport);
                    }
                    reader.endArray();
                } else if (name.equals("log_time")) {
                    logTime = dateTimeFormat.parse(reader.nextString());
                } else {
                    reader.skipValue();
                }
            }
            reader.endObject();
            reader.close();
        } catch (IOException e) {
            throw new IllegalStateException(e.getMessage());
        }

        for (BsimportActiveOrdersFormat key : values) {
             context.write(logTime,
                 key.getOrderID(),
                 key.getStop(),
                 key.getShows(),
                 key.getClicks(),
                 key.getSpentUnits(),
                 key.getCost(),
                 key.getCostCur(),
                 dateTimeFormat.parse(key.getUpdateTime())
             );
        }
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}
