package ru.yandex.market.logshatter.parser.direct;

import ru.yandex.market.clickhouse.ddl.Column;
import ru.yandex.market.clickhouse.ddl.ColumnType;
import ru.yandex.market.clickhouse.ddl.engine.EngineType;
import ru.yandex.market.clickhouse.ddl.engine.MergeTree;
import ru.yandex.market.logshatter.parser.LogParser;
import ru.yandex.market.logshatter.parser.ParserContext;
import ru.yandex.market.logshatter.parser.TableDescription;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Date;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonElement;

public class CampaignBalanceLogParser implements LogParser {

    //тут описывается структура таблицы и задаются переменные для парсера

    public static final String DATE_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final SimpleDateFormat dateFormatJava = new SimpleDateFormat(DATE_PATTERN);

    public static final Column DATE_COLUMN = new Column("log_date", ColumnType.Date);
    public static final Column TIMESTAMP_COLUMN = new Column("log_time", ColumnType.DateTime);

    private static final String semplName = "intHash64(reqid)";

    public static List<String> additionalPrimaryKeys = Arrays.asList(semplName);
    private static final List<String> primaryKeys = getPrimary();
    private static final String partName = DATE_COLUMN.getName();

    private static final MergeTree DEFAULT_ENGINE = new MergeTree("toYYYYMM(" + partName + ")", primaryKeys, semplName, 8192);

    private static List<Column> columnsStruct = new ArrayList<Column>(Arrays.asList(
        new Column("service", ColumnType.String),
        new Column("method", ColumnType.String),
        new Column("host", ColumnType.String),
        new Column("reqid", ColumnType.UInt64),
        new Column("ip", ColumnType.String),
        new Column("tid", ColumnType.String),
        new Column("cid", ColumnType.UInt64),
        new Column("ClientID", ColumnType.UInt64),
        new Column("type", ColumnType.String),
        new Column("currency", ColumnType.String),
        new Column("sum", ColumnType.Int64),
        new Column("sum_delta", ColumnType.Int64),
        new Column("sum_balance", ColumnType.Int64)
    ));

    public static TableDescription create(EngineType engineType, List<Column> columns) {
        List<Column> allColumns = new ArrayList<>(columns.size() + 2);
        allColumns.add(DATE_COLUMN);
        allColumns.add(TIMESTAMP_COLUMN);
        allColumns.addAll(columns);
        return new TableDescription(allColumns, engineType);
    }

    private static final TableDescription TABLE_DESCRIPTION = create(DEFAULT_ENGINE, columnsStruct);

    private static final Gson gson = new Gson();

    private static List<String> getPrimary() {
        List<String> aggrKeys = new ArrayList<>();
        aggrKeys.add(DATE_COLUMN.getName());
        aggrKeys.add(TIMESTAMP_COLUMN.getName());
        aggrKeys.addAll(additionalPrimaryKeys);
        return aggrKeys;
    }

    @Override
    public void parse(String line, ParserContext context) throws Exception {
        JsonObject record = gson.fromJson(line, JsonObject.class);

        String logType = record.get("log_type").getAsString();
        if (!logType.equals("campaign_balance")) {
            return;
        }
        Date date = dateFormatJava.parse(record.get("log_time").getAsString());
        String service = record.get("service").getAsString();
        String method = record.get("method").getAsString();
        String host = record.get("log_hostname").getAsString();
        long reqid = record.get("reqid").getAsLong();
        String ip = record.get("ip").getAsString();

        JsonObject data = record.get("data").getAsJsonObject();
        String type = data.get("type").getAsString();
        String tid = data.get("tid").getAsString();
        long cid = data.get("cid").getAsLong();
        long ClientID = data.get("ClientID").getAsLong();
        String currency = data.get("currency").getAsString();
        double sum = data.get("sum").getAsDouble() * 1_000_000;
        double sum_delta = data.get("sum_delta").getAsDouble() * 1_000_000;
        double sum_balance = data.get("sum_balance").getAsDouble() * 1_000_000;

        context.write(date, service, method, host, reqid, ip, tid, cid, ClientID,
            type, currency, (long) sum, (long) sum_delta, (long) sum_balance);
    }

    @Override
    public TableDescription getTableDescription() {
        return TABLE_DESCRIPTION;
    }
}